/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#ifndef _mom_pbe_c_h_
#define _mom_pbe_c_h_

extern "C" {

#include <stdio.h>
#include <stdarg.h>

#include <rpc/types.h>
#include <rpc/xdr.h>

#include <mom/pres_c.h>
#include <mom/compiler.h>
}

struct be_flags {
	char *input;
	char *output;
	char *header;
	char *prefix;
	int system_header;
	int no_timestamp;
	int no_mu_stubs;
	int no_includes;
};

be_flags get_default_be_flags();
be_flags be_args(int argc, char **argv, const be_flags &def_flags,
		 char *info=0);

/* Use these defines to _test_ assumption masks... */

/* Assume other side has same architecture/environment -
   no data format conversions necessary.  */
#define RPCA_SAME_ARCH		0x00000001

/* Assume the other side is on the same machine,
   which means we can use registerized parameters.
   Implies RPCA_SAME_ARCH.  */
#define RPCA_SAME_NODE		0x00000002

/* Assume the other side is trusted for security
   (but not necessarily for integrity).  */
#define RPCA_SECURE		0x00000004

/* Assume we fully trust the other side.
   RPCA_SECURE must also be set.  */
#define RPCA_TRUSTED		0x00000008

/* Assume we are fully dependent on the other side's continued existence -
   we would never be able to keep running sensibly if the other side fails,
   even if it doesn't directly damage our domain in doing so.
   Put another way, we assume that the target object will never go away
   until all references to it are gone - if it does, we die too.
   RPCA_TRUSTED must also be set.  */
#define RPCA_DEPEND		0x00000010

/* Assume client and server are in the same protection domain.
   RPCA_TRUSTED must also be set.  */
#define RPCA_SAME_DOMAIN	0x00000020

/* Use these defines to _build_ assumption masks... */
#define RPCM_SAME_ARCH		(RPCA_SAME_ARCH)
#define RPCM_SAME_NODE		(RPCA_SAME_NODE | RPCM_SAME_ARCH)
#define RPCM_SECURE		(RPCA_SECURE)
#define RPCM_TRUSTED		(RPCA_TRUSTED | RPCM_SECURE)
#define RPCM_DEPEND		(RPCA_DEPEND | RPCM_TRUSTED)
#define RPCM_SAME_DOMAIN	(RPCA_SAME_DOMAIN | RPCM_TRUSTED)


/* XXX machdep */
#define IPC_MAX_REGS	8
#define IPC_MAX_RIGHTS	1
#define IPC_MAX_BYTES	4096

#define IPC_REG_SIZE	32

// This is a little helper function...
cast_expr make_case_value_expr(mint_const mint_literal);

struct inline_state;
struct mu_state;
struct target_mu_state;
struct client_mu_state;

struct functor
{
	virtual void func(mu_state *must) = 0;
};

struct mu_inline_struct_union_error_functor : public functor
{
	virtual void func(mu_state *must);
};

struct inline_state
{
	/* Build an expression to access slot 'slot' of the inline_state 'ist'.
	   Also return the C type of the specified slot.  */
	virtual void slot_access(int slot, cast_expr *out_expr, cast_type *out_type) = 0;
};

struct func_inline_state : public inline_state
{
	/* The function type itself.  */
	cast_func_type *func_type;

	inline func_inline_state(cast_func_type *_cfunct)
		{ func_type = _cfunct; }

	virtual void slot_access(int slot, cast_expr *out_expr, cast_type *out_type);
};

struct struct_inline_state : public inline_state
{
	/* The structure type itself - must by CAST_TYPE_STRUCT.  */
	cast_struct_type *struct_type;

	/* This is the expression representing the struct variable.  */
	cast_expr var_expr;

	inline struct_inline_state(cast_struct_type *_struct_type, cast_expr _var_expr)
		{ struct_type = _struct_type; var_expr = _var_expr; }

	virtual void slot_access(int slot, cast_expr *out_expr, cast_type *out_type);
};


/* Used by mu_decode_switch */
struct decode_switch_case
{
	pres_c_inline inl;
	pres_c_server_func *server_func;
	pres_c_server_skel *server_skel;
};


typedef int mu_state_op;

const int MUST_DECODE = 0x1;
const int MUST_ENCODE = 0x2;
const int MUST_ALLOCATE = 0x4;
const int MUST_DEALLOCATE = 0x8;

/* Base class for capturing memory allocation state. */
struct mu_memory_allocator_state
{
};

/* `MAX_STUB_RECURSIVE_INLINE_DEPTH' determines how many times the back end may
   recursively inline the body of any stub.
   
   If this value is zero, then the back end will never inline any stub; it will
   instead call every stub through a function call.
   
   If MAX_STUB_RECURSIVE_INLINE_DEPTH is one, then the back end will inline up
   to one level of every stub.  For example, if while inlining stub `X' the
   back end needs to invoke stub `X' (i.e., make a recursive call), then the
   recursive invocation of `X' will be a function call, not an inline.  Note
   that the MAX_STUB_RECURSIVE_INLINE_DEPTH limit only applies to `X' within
   `X' --- the back end may freely inline *other* stubs within `X' (unless
   doing so would violate some other limit, of course).
   
   Greater values mean that the back end will recursively inline stubs to
   greater depths.
    */
const int MAX_STUB_RECURSIVE_INLINE_DEPTH = 1;

/* This is a generic transport from nodes lower in the pres_c tree to
   nodes higher in the pres_c tree.
   (This is useful for passing information between nodes) */
struct mu_state_arglist {
	struct argument {
		char     *name;
		cast_expr cexpr;
		cast_type ctype;
		argument *next;
	} *arg_list;

	void add(char *nm);
	void remove(char *nm);
	
	/* the return value is 0 for success */
	int getargs(char *nm, cast_expr *expr, cast_type *type);
	int setargs(char *nm, cast_expr expr, cast_type type);
	mu_state_arglist() {arg_list=NULL;};
};
	
/*
Two base classes for code generation exist.
This class represents the general case - 
navigation through the pres_c file.
This class is abstract.
*/
struct mu_state
{
	/*
	  This is a pointer to the pres_c structure
	  which is generated by the front end.
	  mu_state iterates over this structure.
	  */
	pres_c_1 *pres;
	
	/* The `stub_inline_depth' array keeps track of the "depth" at which we
	   are inlining each stub described in `pres'.  Every time we start
	   inlining (i.e., producing the C code within) the body of a stub, we
	   increment the counter associated with that stub.  When we finish, we
	   decrement the counter.  Certain functions such as `mu_mapping_stub'
	   examine these counters in order to determine whether we should
	   inline the body of a stub or produce a call to the stub.
	   */
	int *stub_inline_depth;
	char *which_stub;
	
	/* Operation to be performed by the stub.  */
        mu_state_op op;

	/*
	 * The direction of the parameter for which we are currently processing
	 * data: `in', `inout', `out', `return', or `unknown'.
	 */
	pres_c_direction current_param_dir;
	
	// According to Nathan Dykman (6/10/96), this isn't used 
	int assumptions;
	
	/*
	 * The no-double-swap optimization. <Gnats #61>.
	 * 1 if currently packing several chars into a short or an int
	 * for discriminating upon.  This is done in hash_const.cc when
	 * doing a switch on a string.  See hash_const.cc for more details.
	 */
	int now_packing;	
	
	/* List of pres_c_def's for which we are doing inline marshaling.
	   This is used as a stack to detect loops
	   that we have to break with calls to out-of-line recursive stubs.  */
	/*...*/
	/* XXX for starters, don't bother doing any inline marshaling of pres_c_def's
	   (only pres_c_decl's).  */

	/* C marshal/unmarshal code block we're currently building,
	   null if none so far.  */
	cast_stmt c_block;
	
	/* When we're marshaling/unmarshaling the length of an array,
	   'array_def' points to the MINT array definition; otherwise
	   it's null. */
	mint_array_def *array_def;
	
	/*
	 * These are the CAST expressions for presented security identifiers
	 * (SIDs).  The base `mu_state' class knows when to set/clear these
	 * slots, but it does not know what to do with the actual expressions.
	 * Individual BE's must know what to do when any SID must be managed.
	 */
	cast_expr client_sid_cexpr;
	cast_expr server_sid_cexpr;

	mu_state_arglist *arglist;
	
	/*********************************************************************/
	
	mu_state(pres_c_1 *_pres, mu_state_op _op, int _assumptions, char *which);
	mu_state(const mu_state &must);
	/* clone & another operations are implementation specific.
	   They are used to create iterators over
	   the pres_c structure which gather info
	   to be used inthe actual code gen phase. 
	   Nate Dykman said it's like trace scheduling,
	   but that seems like a stretch...
	   */
	virtual mu_state *clone() = 0;
	virtual mu_state *another(mu_state_op _op) = 0;
	virtual ~mu_state();

	/* Get an ASCII string describing the current operation -
	   generally used when constructing magic symbol names to spit out in generated code.  */
	virtual char *get_mem_name()
	{
		switch (op & (MUST_ALLOCATE | MUST_DEALLOCATE)) {
		case MUST_ALLOCATE:	return "alloc";
		case MUST_DEALLOCATE:	return "dealloc";
		default:		return "buffer";
		}
	}
	virtual char *get_buf_name()
	{ 
		switch (op & (MUST_ENCODE | MUST_DECODE)) {
		case MUST_ENCODE:	return "encode";
		case MUST_DECODE:	return "decode";
		default:		return "memory";
		}
	}
	/* Returns the name of this backend,
	   for embedding in marshaling macro names and such.  */
	virtual char *get_encode_name() = 0;
	virtual char *get_be_name() = 0;
	virtual char *get_which_stub();
	
	/* This should return the most characters an encoding can
	   pack to switch on for discriminators
	   */
	virtual int get_most_packable() = 0;
	
	/* These functions create statements and variables.
	   The reason you can't just add a variable as a statement
	   is that they must be shuffled forward to
	   the beginning of a code block
	   (C limitation, no 'inline' variables)
	   */

	void add_var(char *name, cast_type type, cast_storage_class sc = CAST_SC_NONE);
	void add_stmt(cast_stmt st);
	
	/* This function adds a line of code with no indentation.  It is
	   currently used for "#if"'s, "#else"'s and "#endif"'s. */
	
	void add_direct_code(char *code_string);
	
	/* Create a temporary variable of the given type in the current
	   c_block, and return an expression representing its lvalue.
	   'tag' is a short constant string mangled into the name that
	   doesn't actually matter;
	   it's only to make the generated code more understandable.  */
	cast_expr add_temp_var(const char *tag, cast_type type,
				cast_storage_class sc = CAST_SC_NONE);

	// This handles simple data types, and is very transport specific
	virtual void mu_mapping_simple(cast_expr expr, cast_type ctype, mint_ref itype) = 0;
	
	// This just calls mu_mapping_simple, but can be overridden if desired
	virtual void mu_array_elem(cast_expr elem_expr, cast_type elem_ctype,
				   mint_ref elem_itype, pres_c_mapping elem_mapping,
				   unsigned long len_min, unsigned long len_max);

	// This just steps thru the array and calls mu_array_elem
	// It should probably be overridden to make it more efficient
	virtual void mu_array(
		cast_expr array_expr, cast_type array_ctype,
		pres_c_allocation *array_alloc,
		cast_type elem_ctype, mint_ref elem_itype,
		pres_c_mapping elem_map,
		cast_expr len_expr, cast_type len_ctype,
		unsigned long len_min, unsigned long len_max);
  
	// This returns nil, if it's not a fixed size, or a (constant) size
	// expression otherwise
	virtual cast_expr mu_get_sizeof(mint_ref itype,
					cast_type ctype,
					pres_c_mapping map,
					int *size,
					int *align_bits);
  
	// This returns a #if (0), #else (1), or #endif (2) for bit-translation checking
	virtual cast_stmt mu_bit_translation_necessary(int);

	// The creates a union handler.  The functor should be the demux function
	virtual void mu_union_case(functor *f);
	virtual void mu_union(functor *f);

	virtual void mu_pointer_alloc(cast_expr expr, cast_type target_type,
				      cast_expr count_expr, pres_c_allocation *alloc);
	virtual void mu_pointer_free(cast_expr expr, cast_type target_type,
				     cast_expr lexpr, pres_c_allocation *alloc);

	virtual void mu_mapping_direct(cast_expr expr, cast_type ctype, mint_ref itype);
	virtual void mu_mapping_direction(cast_expr expr,
					  cast_type ctype,
					  mint_ref itype,
					  pres_c_mapping_direction *dir_map);
	virtual void mu_mapping_fixed_array(
		cast_expr expr,
		cast_type ctype,
		mint_ref itype,
		pres_c_mapping_fixed_array *array_mapping);
	virtual void mu_mapping_struct(cast_expr expr, cast_type ctype, mint_ref itype,
				       pres_c_inline inl);
	virtual void mu_mapping_stub_inline(cast_expr expr, cast_type ctype, mint_ref itype,
					    pres_c_mapping map);
	virtual void mu_mapping_stub_call(cast_expr expr, cast_type ctype, mint_ref itype,
					  pres_c_mapping map);
	virtual void mu_mapping_stub(cast_expr expr, cast_type ctype, mint_ref itype,
				     pres_c_mapping map);
	virtual void mu_mapping_pointer(cast_expr expr, cast_type ctype, mint_ref itype,
					pres_c_mapping_pointer *pmap);
	virtual void mu_mapping_xlate(cast_expr expr, cast_type ctype, mint_ref itype,
				      pres_c_mapping_xlate *xmap);
	virtual void mu_mapping_reference_get_attributes(
		mint_ref itype, pres_c_mapping_reference *rmap,
		int *ref_count_adjust, int *mark_for_cleanup);
	virtual void mu_mapping_reference(cast_expr expr, cast_type ctype, mint_ref itype,
					  pres_c_mapping_reference *rmap);
	virtual void mu_mapping_type_tag(cast_expr expr, cast_type ctype, mint_ref itype);
	
	virtual void mu_mapping_terminated_array(cast_expr expr,
						 cast_type ctype,
						 mint_ref itype,
						 pres_c_mapping_terminated_array *tmap);
	virtual void mu_mapping_optional_pointer(cast_expr expr,
						 cast_type ctype,
						 mint_ref itype,
						 pres_c_mapping_optional_pointer *ptr_map);
	virtual void mu_mapping_system_exception(cast_expr expr,
						 cast_type ctype,
						 mint_ref itype);
	virtual void mu_mapping_string(cast_expr ptr,
				       cast_type ptr_ctype,
				       mint_array_def *arr,
				       pres_c_allocation *mem_alloc) = 0;
	
	virtual void mu_mapping_sid(cast_expr expr,
				    cast_type ctype,
				    mint_ref itype,
				    pres_c_mapping_sid *sid_map);
	
	virtual void mu_mapping_argument(cast_expr expr,
					 cast_type ctype,
					 mint_ref itype,
					 pres_c_mapping map);
	
	virtual void mu_mapping_message_attribute(cast_expr expr,
						  cast_type ctype,
						  mint_ref itype,
						  pres_c_mapping_message_attribute *attr_map);
	
	virtual void mu_mapping(cast_expr expr,
				cast_type ctype,
				mint_ref itype,
				pres_c_mapping map);

	virtual void mu_inline_message_attribute(inline_state *ist,
						 mint_ref itype,
						 pres_c_inline inl);
	
	void mu_inline_struct_union(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_virtual_union(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_void_union(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_collapsed_union(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_typed(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_counted_array(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_terminated_array(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_atom(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_struct(inline_state *ist, mint_ref itype, pres_c_inline inl);
        /* volkmar: has to be virtual - I have to overload to reorder!!! */
	virtual void mu_inline_func_params_struct(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_xlate(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_cond(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline_assign(inline_state *ist, mint_ref itype, pres_c_inline inl);
	void mu_inline(inline_state *ist, mint_ref itype, pres_c_inline inl);

	void mu_func_params(cast_ref cfunc_idx, mint_ref itype, pres_c_inline inl);

	/*
	  These functions handle constant values present in 
	  the pres_c structure.
	  Constants usually appear on the server stubs,
	  as a constant value,
	  which is used to determine if the client is calling the right server.
	  */
/*****
	virtual void mu_decode_const(mint_const *darray, int darray_len, mint_ref n_r,
				     cast_expr cookie_var);
*****/
	virtual void mu_encode_const(mint_const iconst, mint_ref itype);
	virtual void mu_const(mint_const iconst, mint_ref itype);

        virtual void mu_hash_const(mint_const *domain,	// the 'domain' of inputs
				   int domain_size,	// the # of inputs
				   mint_ref const_type,	// the type of the inputs
				   cast_expr var);	// the variable to hash to
	/* The "new and improved" `mu_hash_const'. */
	virtual void mu_discriminate(mint_const *domain,
				     mint_ref domain_itype,
				     int domain_size,
				     functor **success_functors,
				     functor *failure_functor,
				     cast_type discrim_ctype, /* may be void */
				     cast_expr discrim_expr); /* may be void */
	
	/*
	  These functions are responsible for creating the server stub. For each
	  client function, a case inside a switch statement is made. Calls to
	  the server stub include a parameter identifying which server function
	  is to be called. These functions create that code, and are the
	  starting point for server stub generation.
	  */

	virtual void mu_decode_switch(decode_switch_case *darray, int darray_len, mint_ref n_r);
	virtual void mu_server_func_target(pres_c_server_func *sfunc);
	virtual void mu_server_func_client(pres_c_server_func *sfunc);
	virtual void mu_server_func_reply(pres_c_server_func *sfunc, pres_c_server_skel *sskel);
	virtual void mu_server_func_alloc_param(cast_func_param *param);
	
	virtual void mu_server_func(pres_c_inline inl, mint_ref tn_r,
				    pres_c_server_func *sfunc,
				    pres_c_server_skel *sskel);
	
	/* This function gives one an appropriate "target" `mu_state' object,
	   which is necessary for marshaling/unmarshaling references to objects
	   that are the "targets" of RPC or RMI (remote method invocation).
	   
	   This method must be specialized for every class that is derived from
	   `mu_state', because every transport deals with object references in
	   a different way. */
	virtual target_mu_state *mu_make_target_mu_state(pres_c_1 *pres,
							 mu_state_op op,
							 int assumptions,
							 char *which_stub) = 0;
	/* Also for the "client" object. */
	virtual client_mu_state *mu_make_client_mu_state(pres_c_1 *pres,
							 mu_state_op op,
							 int assumptions,
							 char *which_stub) = 0;
	
	/*
	  Provides a means of adding code at the end of the block to do various
	  cleanup and other tasks.
	  */
	virtual void mu_end();
	
	/*
	  These methods handle the building blocks of memory allocation:
	  "globs" and "chunks."  A glob is a contiguous region of memory that
	  has a known maximum size.  A chunk is a contiguous region of memory
	  within a glob; a chunk has an offset within its glob and a fixed
	  size.
	  
	  "Globs" and "chunks" may be realized in different ways by different
	  back ends.  The default implementations of these methods are no-ops.
	  Individual back ends are expected to override these methods.
	  */
	virtual void new_glob();
	virtual void make_glob();
	virtual void break_glob();
	virtual void end_glob();
	
	virtual void glob_grow(int amount);
	virtual void glob_prim(int needed_align_bits, int prim_size);
	
	virtual void new_chunk();
	virtual void new_chunk_align(int needed_bits);
	virtual void break_chunk();
	virtual void end_chunk();
	
	/*
	  New methods to get and set the state of the memory allocator.  This
	  allows us to do arbitrary rewinds, etc.
	  */
	virtual mu_memory_allocator_state *memory_allocator_state();
	virtual void set_memory_allocator_state(mu_memory_allocator_state *);
	virtual char *get_deallocator(pres_c_allocation *alloc);
	virtual char *get_allocator(pres_c_allocation *alloc);
	virtual cast_stmt make_error(char *err_val);
	// This will be called before we begin marshaling or unmarshaling parameters
	virtual void mu_prefix_params(void);
};



/* Basic stub generation functions */
void w_header_prologue(be_flags *flags, pres_c_1 *pres);
void w_header_epilogue(be_flags *flags, pres_c_1 *pres);
void w_header_includes(pres_c_1 *pres);
void w_source_includes(char *header, char *prefix, int system);

void w_stub(pres_c_1 *pres, int idx);
void w_marshal_stub(pres_c_1 *pres, int idx);
void w_unmarshal_stub(pres_c_1 *pres, int idx);
void w_client_stub(pres_c_1 *pres, int stub_idx);
void w_server_skel(pres_c_1 *pres, int stub_idx);
void w_send_stub(pres_c_1 *pres, int stub_idx);

void do_main_output(pres_c_1 *pres);

extern cast_expr emergency_return_value;

/* This is for cleaning up the glop on the interface unions */
void remove_idl_and_interface_ids(
	pres_c_1 *pres,
	mint_ref in_itype, pres_c_inline in_inline,
	mint_ref *out_itype, pres_c_inline *out_inline);

void remove_operation_id(
	pres_c_1 *pres,
	mint_ref in_itype, pres_c_inline in_inline,
	mint_ref *out_itype, pres_c_inline *out_inline);


#endif /* _mom_pbe_c_h_ */

/* End of file. */


