/*
 * Copyright (c) 1995, 1996 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#ifdef RPC_HDR
%#ifndef _flick_cast_h
%#define _flick_cast_h
%
%#include <rpc/types.h>
%#include <rpc/xdr.h>
%#include <mom/aoi.h>
#endif

/* Note that a cast represents an abstract syntax tree representation,
   not a parse tree: it's somewhat higher-level than a parse tree in various respects:
   * Type constructors are represented here forwards,
     rather than backwards as they would be in a parse tree.
   * There is no representation of grouping parentheses;
     all grouping is represented implicitly by the AST nodes.
   * There is no concept of declaring multiple identifiers in a single declaration.
   * The -> operator is not present; (*foo).bar is used instead.
   * The array index operator is not present; *((array)+(index)) is used instead.

   However, it's still a tree representation, not a graph:
   cross-references are by name, not by pointer.

   Features of C that currently aren't supported at all:
   * Bit fields in structures.

   Cleanups due at some point:
   * Get rid of bogus pointer indirections when MOM can fix it.
   * Get rid of cast_pointer_type structure
   * Rename cast_block to cast_compound?
   * Rename cast_def* to cast_decl*
*/

/* These are kludges to get around the limitations of current C language mappings.  */
typedef struct cast_type_u *cast_type;
typedef struct cast_expr_u *cast_expr;
typedef struct cast_stmt_u *cast_stmt;
typedef struct cast_init_u *cast_init;
typedef struct cast_def cast_scope<>;


/***** TYPE CONSTRUCTORS *****/

/* This is used for both struct slots and union cases.  */
struct cast_field
{
	string		name<>;
	cast_type	type;
};

enum cast_primitive_kind
{
	CAST_PRIM_CHAR		= 1,
	CAST_PRIM_INT		= 2,
	CAST_PRIM_FLOAT		= 3,
	CAST_PRIM_DOUBLE	= 4
};
typedef string alias<>;
typedef unsigned cast_primitive_modifier;
const CAST_MOD_SIGNED		= 0x0001;
const CAST_MOD_UNSIGNED		= 0x0002;
const CAST_MOD_SHORT		= 0x0010;
const CAST_MOD_LONG		= 0x0020;
const CAST_MOD_LONG_LONG	= 0x0040;
struct cast_primitive_type
{
	cast_primitive_kind	kind;
	cast_primitive_modifier	mod;
	alias			*name;
};

typedef cast_field cast_struct_slot;

struct cast_enum_field
{
	/* The name of the slot */
	string name<>;

	/* The value (should probably be a literal primitive */
	cast_expr val;
};

struct cast_enum_type
{
	/* Enum tag. */
	string name<>;

	/* value and name of each slot */
	cast_enum_field slots<>;
};	

struct cast_struct_type
{
	/* Structure tag.  */
	string name<>;

	/* This defines the slots of the structure, in order.  */
	cast_struct_slot slots<>;
};

struct cast_array_type
{
	/* Constant expression that determines the array's length.
	   Can be omitted if outside of anything or at the end of a structure;
	   means array is variable-length.  */
	cast_expr		length;

	/* Type of each element.  */
	cast_type		element_type;
};

typedef cast_field cast_union_case;
struct cast_union_type
{
	/* Union tag.  */
	string name<>;

	cast_union_case cases<>;
};

struct cast_pointer_type
{
	/* What the pointer points to.  */
	cast_type		target;
};

typedef cast_field cast_func_param;
struct cast_func_type
{
	cast_func_param params<>;

	cast_type return_type;
};

typedef int cast_type_qualifier;
const CAST_TQ_CONST		= 0x01;
const CAST_TQ_VOLATILE		= 0x02;
struct cast_qualified_type
{
	cast_type_qualifier qual;
	cast_type actual;
};

enum cast_type_kind
{
	CAST_TYPE_NAME		= 1,
	CAST_TYPE_PRIMITIVE	= 2,
	CAST_TYPE_STRUCT	= 3,
	CAST_TYPE_UNION		= 4,
	CAST_TYPE_ARRAY		= 5,
	CAST_TYPE_POINTER	= 6,
	CAST_TYPE_FUNCTION	= 7,
	CAST_TYPE_ENUM		= 8,
	CAST_TYPE_STRUCT_NAME	= 9,
	CAST_TYPE_UNION_NAME	= 10,
	CAST_TYPE_ENUM_NAME	= 11,
	CAST_TYPE_VOID		= 12,
	CAST_TYPE_QUALIFIED	= 13
};
union cast_type_u
switch (cast_type_kind kind)
{
	case CAST_TYPE_NAME:		string			name<>;
	case CAST_TYPE_PRIMITIVE:	cast_primitive_type	primitive_type;
	case CAST_TYPE_STRUCT:		cast_struct_type	struct_type;
	case CAST_TYPE_UNION:		cast_union_type		union_type;
	case CAST_TYPE_POINTER:		cast_pointer_type	pointer_type;
	case CAST_TYPE_ARRAY:		cast_array_type		array_type;
	case CAST_TYPE_FUNCTION:	cast_func_type		func_type;
	case CAST_TYPE_ENUM:		cast_enum_type		enum_type;
	case CAST_TYPE_STRUCT_NAME:	string			struct_name<>;
	case CAST_TYPE_UNION_NAME:	string			union_name<>;
	case CAST_TYPE_ENUM_NAME:	string			enum_name<>;
	case CAST_TYPE_VOID:		void;
	case CAST_TYPE_QUALIFIED:	cast_qualified_type	qualified;
};


/***** EXPRESSIONS *****/

union cast_lit_prim_u
switch (cast_primitive_kind kind)
{
	case CAST_PRIM_CHAR:	char		c;
	case CAST_PRIM_INT:	long		i; /* XXX long long */
	case CAST_PRIM_FLOAT:	float		f;
	case CAST_PRIM_DOUBLE:	double		d; /* XXX long double */
};
struct cast_lit_prim
{
	cast_primitive_modifier	mod;
	cast_lit_prim_u		u;
};

struct cast_expr_call
{
	cast_expr	func;
	cast_expr	params<>;
};

struct cast_expr_sel
{
	cast_expr	var;
	string		member<>;
};

enum cast_unary_op
{
	CAST_UNARY_DEREF	= 1,	/* * (unary) */
	CAST_UNARY_ADDR		= 2,	/* & (unary) */
	CAST_UNARY_NEG		= 3,	/* - (unary) */
	CAST_UNARY_LNOT		= 4,	/* ! (unary) */
	CAST_UNARY_BNOT		= 5,	/* ~ (unary) */
	CAST_UNARY_PRE_INC	= 6,	/* ++foo */
	CAST_UNARY_PRE_DEC	= 7,	/* --foo */
	CAST_UNARY_POST_INC	= 8,	/* foo++ */
	CAST_UNARY_POST_DEC	= 9	/* foo-- */
};
struct cast_unary_expr
{
	cast_unary_op	op;
	cast_expr	expr;
};

struct cast_expr_cast
{
	/* Expression to cast.  */
	cast_expr	expr;

	/* Type to cast it to.  */
	cast_type	type;
};

enum cast_binary_op
{
	CAST_BINARY_MUL		= 1,	/* * (binary) */
	CAST_BINARY_DIV		= 2,	/* / */
	CAST_BINARY_MOD		= 3,	/* % */
	CAST_BINARY_ADD		= 4,	/* + (binary) */
	CAST_BINARY_SUB		= 5,	/* - (binary) */
	CAST_BINARY_SHL		= 6,	/* << */
	CAST_BINARY_SHR		= 7,	/* >> */

	CAST_BINARY_LT		= 8,	/* < */
	CAST_BINARY_GT		= 9,	/* > */
	CAST_BINARY_LE		= 10,	/* <= */
	CAST_BINARY_GE		= 11,	/* >= */
	CAST_BINARY_EQ		= 12,	/* == */
	CAST_BINARY_NE		= 13,	/* != */

	CAST_BINARY_BAND	= 14,	/* & */
	CAST_BINARY_BXOR	= 15,	/* ^ */
	CAST_BINARY_BOR		= 16,	/* | */

	CAST_BINARY_LAND	= 17,	/* && */
	CAST_BINARY_LOR		= 18,	/* || */

	CAST_BINARY_ASSIGN	= 19,	/* = */

	CAST_BINARY_COMMA	= 20	/* , */
};
struct cast_binary_expr
{
	cast_binary_op	op;
	cast_expr	expr[2];
};

struct cast_cond_expr
{
	cast_expr	test;
	cast_expr	true_expr;
	cast_expr	false_expr;
};

enum cast_expr_kind
{
	CAST_EXPR_NAME		= 1,
	CAST_EXPR_LIT_PRIM	= 2,
	CAST_EXPR_LIT_STRING	= 3,
	CAST_EXPR_CALL		= 4,	/* <expr>(<params>) */
	CAST_EXPR_SEL		= 5,	/* <struct/union>.<member> */
	CAST_EXPR_UNARY		= 6,
	CAST_EXPR_CAST		= 7,	/* (<type>) */
	CAST_EXPR_SIZEOF_EXPR	= 8,	/* sizeof(<expr>) */
	CAST_EXPR_SIZEOF_TYPE	= 9,	/* sizeof(<type>) */
	CAST_EXPR_BINARY	= 10,
	CAST_EXPR_OP_ASSIGN	= 11,	/* <op>= */
	CAST_EXPR_COND		= 12	/* ? : */
};
union cast_expr_u
switch (cast_expr_kind kind)
{
	case CAST_EXPR_NAME:		string			name<>;
	case CAST_EXPR_LIT_PRIM:	cast_lit_prim		lit_prim;
	case CAST_EXPR_LIT_STRING:	string			lit_string<>;
	case CAST_EXPR_CALL:		cast_expr_call		call;
	case CAST_EXPR_SEL:		cast_expr_sel		sel;

	case CAST_EXPR_UNARY:		cast_unary_expr		unary;
	case CAST_EXPR_CAST:		cast_expr_cast		cast;
	case CAST_EXPR_SIZEOF_EXPR:	cast_expr		sizeof_expr;
	case CAST_EXPR_SIZEOF_TYPE:	cast_type		sizeof_type;

	case CAST_EXPR_BINARY:		cast_binary_expr	binary;
	case CAST_EXPR_OP_ASSIGN:	cast_binary_expr	op_assign;
	case CAST_EXPR_COND:		cast_cond_expr		cond;
};

/***** STATEMENTS *****/

struct cast_block
{
	/* Variable declarations and such at beginning of block.  */
	cast_scope		scope;

	/* Statements following them.  */
	cast_stmt		stmts<>;
};

struct cast_if
{
	cast_expr		test;
	cast_stmt		true_stmt;
	cast_stmt		false_stmt;	/* optional */
};

struct cast_while
{
	cast_expr		test;
	cast_stmt		stmt;
};

struct cast_for
{
	cast_expr		init;	/* optional */
	cast_expr		test;	/* optional */
	cast_expr		iter;	/* optional */
	cast_stmt		stmt;
};

struct cast_switch
{
	cast_expr		test;
	cast_stmt		stmt;
};

struct cast_label
{
	string			label<>;
	cast_stmt		stmt;
};
struct cast_case
{
	cast_expr		label;
	cast_stmt		stmt;
};

enum cast_stmt_kind
{
	CAST_STMT_EXPR		= 1,
	CAST_STMT_BLOCK		= 2,
	CAST_STMT_IF		= 3,
	CAST_STMT_WHILE		= 4,
	CAST_STMT_DO_WHILE	= 5,
	CAST_STMT_FOR		= 6,
	CAST_STMT_SWITCH	= 7,
	CAST_STMT_BREAK		= 8,
	CAST_STMT_CONTINUE	= 9,
	CAST_STMT_GOTO		= 10,
	CAST_STMT_LABEL		= 11,
	CAST_STMT_CASE		= 12,
	CAST_STMT_DEFAULT	= 13,
	CAST_STMT_RETURN	= 14,
        CAST_STMT_TEXT		= 15,
	CAST_STMT_NULL		= 16
};
union cast_stmt_u
switch (cast_stmt_kind kind)
{
	case CAST_STMT_EXPR:		cast_expr		expr;
	case CAST_STMT_BLOCK:		cast_block		block;
	case CAST_STMT_IF:		cast_if			s_if;
	case CAST_STMT_WHILE:		cast_while		s_while;
	case CAST_STMT_DO_WHILE:	cast_while		s_do_while;
	case CAST_STMT_FOR:		cast_for		s_for;
	case CAST_STMT_SWITCH:		cast_switch		s_switch;
	case CAST_STMT_BREAK:		void;
	case CAST_STMT_CONTINUE:	void;
	case CAST_STMT_GOTO:		string			goto_label<>;
	case CAST_STMT_LABEL:		cast_label		s_label;
	case CAST_STMT_CASE:		cast_case		s_case;
	case CAST_STMT_DEFAULT:		cast_stmt		default_stmt;
	case CAST_STMT_RETURN:		cast_expr		return_expr;	/* optional */
	case CAST_STMT_TEXT:		string			text<>;
	case CAST_STMT_NULL:		void;
};



/***** DECLARATIONS *****/

struct cast_func_def
{
	cast_func_type	type;
	cast_block	block;
};

enum cast_init_kind
{
	CAST_INIT_EXPR		= 1,
	CAST_INIT_AGGREGATE	= 2
};
union cast_init_u
switch (cast_init_kind kind)
{
	case CAST_INIT_EXPR:		cast_expr	expr;
	case CAST_INIT_AGGREGATE:	cast_init	subs<>;
};
struct cast_var_def
{
	cast_type	type;

	/* Optional - if present, specifies the variable's initializer.  */
	cast_init	init;
};

struct cast_include
{
	string		filename<>;
	bool		system_only;
};

struct cast_direct
{
	string		code_string<>;
};

enum cast_storage_class
{
	CAST_SC_NONE		= 0,
	CAST_SC_AUTO		= 1,
	CAST_SC_STATIC		= 2,
	CAST_SC_EXTERN		= 3,
	CAST_SC_REGISTER	= 4
};

/* A pres_c_def directly produces a C definition.
   One doesn't necessarily correspond directly to an itype.  */
enum cast_def_kind
{
	CAST_TYPEDEF	= 1,
	CAST_STRUCT	= 2,
	CAST_UNION	= 3,
	CAST_ENUM	= 4,
	CAST_FUNC_DECL	= 6,
	CAST_FUNC_DEF	= 7,
	CAST_VAR_DECL	= 8,
	CAST_VAR_DEF	= 9,
	CAST_DEFINE	= 10,	/* #define XXX */
	CAST_INCLUDE	= 11,	/* #include XXX */
	CAST_DIRECT_CODE = 12
};

union cast_def_u
switch (cast_def_kind kind)
{
	case CAST_TYPEDEF:		cast_type		typedef_type;
	case CAST_STRUCT:		cast_struct_type	struct_type;
	case CAST_UNION:		cast_union_type		union_type;
	case CAST_ENUM:			cast_enum_type		enum_type;
	case CAST_FUNC_DECL:		cast_func_type		func_type;
	case CAST_FUNC_DEF:		cast_func_def		func_def;
	case CAST_VAR_DECL:		cast_type		var_type;
	case CAST_VAR_DEF:		cast_var_def		var_def;
	case CAST_DEFINE:		cast_expr		define_as;
	case CAST_INCLUDE:		cast_include		include;
	case CAST_DIRECT_CODE:		cast_direct		direct;
};

struct cast_def
{
	/* C identifier of type (or whatever) to be declared/defined.
	   Length 0 if none.  */
	string		name<>;

	/* Storage class for the declaration/definition.  */
	cast_storage_class sc;

	/* Description of definition.  */
	cast_def_u	u;

	/* Flag indicating whether or not this was #include'd */
	inclusion	included;
};

typedef cast_scope cast_1;

#ifdef RPC_HDR
%#endif /* _flick_cast_h */
#endif
