/*
 * Copyright (c) 1995, 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 *
 * This file is part of Flick, the Flexible IDL Compiler Kit.
 *
 * Flick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Flick is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Flick; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place #330, Boston, MA 02111, USA.
 */

#ifndef _mom_compiler_h_
#define _mom_compiler_h_

#include <stdio.h>
#include <stdarg.h>
#include <limits.h>

#ifdef __cplusplus
extern "C" {
#endif


#if defined(__GNUC__) && (__GNUC__ >= 2)
#  if __GNUC_MINOR__ >= 7
#    define DECL_NORETURN(decl) decl __attribute__((__noreturn__))
#    define DEFN_NORETURN(decl) decl
#  else
#    if __GNUC_MINOR__ >= 6
#      define DECL_NORETURN(decl) decl __attribute__((noreturn))
#      define DEFN_NORETURN(decl) decl
#    else
#      define DECL_NORETURN(decl) volatile decl
#      define DEFN_NORETURN(decl) volatile decl
#    endif
#  endif
#else
#  define DECL_NORETURN(decl) decl
#  define DEFN_NORETURN(decl) decl
#endif


/* suffixes for default output filenames */
#define AOID_SUFFIX ".aod"
#define PRES_C_SUFFIX ".prc"
#define PRESD_SUFFIX ".prd"


/* main() must provide this variable
   and set it to the value of argv[0].  */
extern char *progname;


/* Print an error message and exit the program.
   Takes a `printf'-style format string and arguments.  */
DECL_NORETURN(void panic(char *format, ...));

/* Print a warning.  Takes a `printf'-style format string and arguments. */
void warn(const char *format, ...);

/* Like the normal allocation routines, but these never return null -
   they just panic if we run out of memory.  */
void *real_mustmalloc(long size, char *file, int line);
void *real_mustcalloc(long size, char *file, int line);
void *real_mustrealloc(void *orig_buf, long new_size, char *file, int line);

#define mustmalloc(size) real_mustmalloc((size), __FILE__, __LINE__)
#define mustcalloc(size) real_mustcalloc((size), __FILE__, __LINE__)
#define mustrealloc(org, size) real_mustrealloc((org), (size), __FILE__, __LINE__)

/* This calls the C preprocessor on the given filename, then returns a
   FILE * to the preprocessed text */
FILE *call_c_preprocessor(char *file, char *flags);

/* "Allocating sprintf" --- allocates exactly the right amount of memory to
   contain the formatted string.  Panics if the allocation fails; never returns
   null. */
char *flick_asprintf(char *fmt, ...);


/* Copy the specified `orig' filename into a new buffer,
   and replace its suffix (the last `.' and everything beyond it)
   with the specified `newsuffix'.  */
char *resuffix(const char *orig, const char *newsuffix);


/* w_indent.c */
void w_indent(int indent);

/* w_print.c */
void w_putc(char c);
void w_vprintf(char *format, va_list vl);
void w_printf(char *format, ...);
void w_set_fh(FILE *out_fh);

/* w_i_print.c */
void w_i_vprintf(int indent, char *format, va_list vl);
void w_i_printf(int indent, char *format, ...);


/* This stuff is for standard command line handling */
enum flag_kind {fk_FLAG, fk_STRING, fk_NUMBER};
typedef enum flag_kind flag_kind;

union flags {
	int flag;
	char *string;
	int number;
};
typedef union flags flags;

struct flags_in {
	char sng;	/* This is the normal flag */
	const char *dbl;	/* This is the --opt flag */
	flag_kind which;
	flags dfault;
	const char *explain;	/* This should provide a string explaning the flag */
};
typedef struct flags_in flags_in;

struct flags_out {
	char *progname;
	flags *flgs;
	int other_count;
	char **other;
	int error;
};
typedef struct flags_out flags_out;

/* This takes a bunch of flags (in the flgs array) and a command line, and builds the flags_out */
flags_out parse_args(int argc, char *argv[], int flag_count, flags_in *flgs);
/* This will print a usage statement automatically */
void print_args_usage(char *progname, int flag_count, flags_in *flgs, char *cl_extra, char *extra);
/* This will pretty-print all the flags - useful for debugging, etc... */
void print_args_flags(flags_out res, int flag_count, flags_in *flgs);

/* This will add the standard flick flags -v/--version, -u/--usage, & -o/--ouput<filename> to the array */
void set_def_flags(flags_in *flgs);
/* This will handle version, usage & errors */
void std_handler(flags_out out, int flag_count, flags_in *flgs, char *cl_info, char *info);

/* These are standard locations for the flags to go */
#define STD_FLAGS 4
#define OUTPUT_FILE_FLAG 2

struct fe_flags {
	char *output;
	char *cpp_flags;
	int nocpp;
	char *input;
};
typedef struct fe_flags fe_flags;
/* Since we don't have an FE library, this is here... */
fe_flags front_end_args(int argc, char **argv, char *info);


#ifdef __cplusplus
}
#endif

#endif /* _mom_compiler_h_ */
