/*
 * Copyright (c) 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation.
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

#ifndef __flick_encode_cdr_h
#define __flick_encode_cdr_h

#include <flick/encode/all.h>
#include <string.h>

extern char flick_is_little_endian;

#define flick_cdr_swap() (((char *)_buf_start)[6]	\
			  != flick_is_little_endian)

#define flick_cdr_unsigned32_size 4
#define flick_cdr_signed32_size 4
#define flick_cdr_swap_unsigned32_size 4
#define flick_cdr_swap_signed32_size 4

/* Begin encoding macros */

/* If you're not using a 2's complement machine, write these yourself... */

#define flick_cdr_encode(_ofs, _data, type)				\
	*(type *)(_buf_current + (_ofs)) = (type)(_data)

#define flick_cdr_encode_char8(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, char)
#define flick_cdr_encode_signed8(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, signed char)
#define flick_cdr_encode_unsigned8(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, unsigned char)
#define flick_cdr_encode_signed16(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, short)
#define flick_cdr_encode_unsigned16(_ofs, _data)			\
	flick_cdr_encode(_ofs, _data, signed short)
#define flick_cdr_encode_signed32(_ofs, _data)				\
	flick_cdr_encode(_ofs, (_data), signed int)
#define flick_cdr_encode_unsigned32(_ofs, _data)			\
	flick_cdr_encode(_ofs, _data, unsigned int)
#define flick_cdr_encode_float32(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, float)
#define flick_cdr_encode_float64(_ofs, _data)				\
	flick_cdr_encode(_ofs, _data, double)
#define flick_cdr_swap_encode_char8(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, char)
#define flick_cdr_swap_encode_signed8(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, signed char)
#define flick_cdr_swap_encode_unsigned8(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, unsigned char)
#define flick_cdr_swap_encode_signed16(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, short)
#define flick_cdr_swap_encode_unsigned16(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, signed short)
#define flick_cdr_swap_encode_signed32(_ofs, _data)	\
	flick_cdr_encode(_ofs, (_data), signed int)
#define flick_cdr_swap_encode_unsigned32(_ofs, _data)			\
	flick_cdr_encode(_ofs, _data, unsigned int)
#define flick_cdr_swap_encode_float32(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, float)
#define flick_cdr_swap_encode_float64(_ofs, _data)	\
	flick_cdr_encode(_ofs, _data, double)

#define flick_cdr_encode_alloced_string(_data, _link, _size)	{	\
	int _temp_iter = 0;						\
        flick_##_link##_encode_new_glob((_size) + 8);			\
	_buf_current =						\
		(void *) ((((unsigned int) _buf_current) + 3)	\
			  & ~3);					\
	while((_data)[_temp_iter]) {					\
		((char *) _buf_current)[_temp_iter + 4] =	\
			((char *) (_data))[_temp_iter];			\
		_temp_iter++;						\
	}								\
	((char *) _buf_current)[_temp_iter + 4] = 0;		\
	*((int *) _buf_current) = _temp_iter + 1;		\
	flick_##_link##_encode_new_chunk(_temp_iter + 5);		\
	flick_##_link##_encode_end_chunk(_temp_iter + 5);		\
	flick_##_link##_encode_end_glob((_size) + 8);			\
}

#define flick_cdr_encode_string(_data, _dealloc, _link, _size) {	\
	flick_cdr_encode_alloced_string(_data, ##_link, _size);		\
	_dealloc((_data));						\
}

#define flick_cdr_encode_longstring(_data, _dealloc, _link) {		\
	int _temp_len, _temp_iter;					\
	for(_temp_len = 0; ((char *) (_data))[_temp_len]; _temp_len++);	\
	flick_##_link##_encode_new_glob(_temp_len + 8);			\
	_buf_current =						\
		(void *) ((((unsigned int) _buf_current + 3)	\
			   & ~3));					\
	flick_##_link##_encode_new_chunk(4);				\
	flick_cdr_encode_unsigned32(0, (_temp_len + 1));		\
	flick_##_link##_encode_end_chunk(4);				\
	flick_##_link##_encode_new_chunk(_temp_len + 1);		\
	for (_temp_iter = 0; _temp_iter <= _temp_len; ++_temp_iter)	\
		*(((char *) _buf_current) + _temp_iter)		\
			= *(((char *) (_data)) + _temp_iter);		\
	flick_##_link##_encode_end_chunk(_temp_len + 1);		\
	_dealloc((_data));						\
	flick_##_link##_encode_end_glob(_temp_len + 8);			\
}

#define flick_cdr_swap_encode_longstring(_data, _dealloc, _link) 	\
        flick_cdr_encode_longstring(_data, _dealloc, _link)
		
#define flick_cdr_encode_bcopy(_ofs, _src, _qty) {	\
	(void) bcopy((void *) (_src),			\
		     (char *) _buf_current + _ofs,	\
		     (_qty));				\
}

/* End encoding section */

/* Constant conversion section */

#if BYTE_ORDER == BIG_ENDIAN

#define flick_cdr_convert_big_endian_to_local_unsigned32(a) (a)

#define flick_cdr_convert_big_endian_to_local_unsigned16(a) (a)

#define flick_cdr_swap_convert_big_endian_to_local_unsigned32(a)	\
       swap_unsigned32(a)

#define flick_cdr_swap_convert_big_endian_to_local_unsigned16(a)	\
       swap_unsigned16(a)
	       
#define flick_mem_convert_big_endian_to_local_unsigned32(a) (a)

#define flick_mem_convert_big_endian_to_local_unsigned16(a) (a)

#else

#define flick_cdr_convert_big_endian_to_local_unsigned32(a)	\
       swap_unsigned32(a)

#define flick_cdr_convert_big_endian_to_local_unsigned16(a)	\
       swap_unsigned16(a)
	       
#define flick_cdr_swap_convert_big_endian_to_local_unsigned32(a) (a)

#define flick_cdr_swap_convert_big_endian_to_local_unsigned16(a) (a)
	       
#define flick_mem_convert_big_endian_to_local_unsigned32(a)	\
       swap_unsigned32(a)

#define flick_mem_convert_big_endian_to_local_unsigned16(a)	\
       swap_unsigned16(a)
	       
#endif

/* End constant conversion section */

/* Decoding macros */

#define flick_cdr_decode(_ofs, _data, type)				\
	(_data) = (*(type *)(_buf_current + (_ofs)))

#define flick_cdr_decode_char8(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, char)
#define flick_cdr_decode_signed8(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, signed char)
#define flick_cdr_decode_unsigned8(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, unsigned char)
#define flick_cdr_decode_signed16(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, short)
#define flick_cdr_decode_unsigned16(_ofs, _data)			\
	flick_cdr_decode(_ofs, _data, signed short)
#define flick_cdr_decode_signed32(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, signed int)
#define flick_cdr_decode_unsigned32(_ofs, _data)			\
	flick_cdr_decode(_ofs, _data, unsigned int)
#define flick_cdr_decode_float32(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, float)
#define flick_cdr_decode_float64(_ofs, _data)				\
	flick_cdr_decode(_ofs, _data, double)

#define flick_cdr_swap_decode8(_ofs, _data, type)			\
	(_data) = (*(type *)(_buf_current + (_ofs)))
#define flick_cdr_swap_decode16(_ofs, _data, type) {			    \
	unsigned short __temp = (*(unsigned short *)(_buf_current + \
						     (_ofs)));		    \
	(_data) = (type)((__temp >> 8) | (__temp << 8));		    \
}
#define flick_cdr_swap_decode32(_ofs, _data, type) {			  \
	unsigned long __temp = (*(unsigned long *)(_buf_current + \
						   (_ofs)));		  \
	(_data) = (type)(((__temp >> 24) & 0x000000FFU) |		  \
		       ((__temp >>  8) & 0x0000FF00U) |			  \
		       ((__temp <<  8) & 0x00FF0000U) |			  \
		       ((__temp << 24) & 0xFF000000U));			  \
}
#define flick_cdr_swap_mem(_ofs, _data, size, type) {			  \
	type __temp = *(type *)(_buf_current + (_ofs));		  \
	char *__vals = (char *)&__temp;					  \
	int __i = 0;							  \
	for (; __i < size; __i++)					  \
		((char *)(_buf_current + (_ofs)))[size - 1 - __i] \
			= __vals[__i];					  \
	flick_cdr_decode(_ofs, _data, type);				  \
}
#define flick_cdr_swap_decode_char8(_ofs, _data)		\
	flick_cdr_swap_decode8(_ofs, _data, char)
#define flick_cdr_swap_decode_signed8(_ofs, _data)			\
	flick_cdr_swap_decode8(_ofs, _data, signed char)
#define flick_cdr_swap_decode_unsigned8(_ofs, _data)		\
	flick_cdr_swap_decode8(_ofs, _data, unsigned char)
#define flick_cdr_swap_decode_signed16(_ofs, _data)		\
	flick_cdr_swap_decode16(_ofs, _data, short)
#define flick_cdr_swap_decode_unsigned16(_ofs, _data)		\
	flick_cdr_swap_decode16(_ofs, _data, signed short)
#define flick_cdr_swap_decode_signed32(_ofs, _data)		\
	flick_cdr_swap_decode32(_ofs, _data, signed int)
#define flick_cdr_swap_decode_unsigned32(_ofs, _data)		\
	flick_cdr_swap_decode32(_ofs, _data, unsigned int)
#define flick_cdr_swap_decode_float32(_ofs, _data)		\
	flick_cdr_swap_mem(_ofs, _data, 4, float)
#define flick_cdr_swap_decode_float64(_ofs, _data)		\
	flick_cdr_swap_mem(_ofs, _data, 8, double)
		
#define flick_cdr_decode_msgptr(_ofs, _ptr) {				\
	(_ptr) = (void *)((char *)_buf_current + (_ofs));	\
}

#define flick_cdr_decode_bcopy(_ofs, _dest, _qty) {	\
	(void) bcopy((char *)_buf_current + _ofs,	\
		     (void *) (_dest),			\
		     (_qty));				\
}

/* There is no flick_cdr_swap_decode_msgptr or flick_cdr_swap_decode_bcopy */

/*
 * The macros for decoding variable-length strings of characters.  These macros
 * assume that when they are invoked, there is no open glob and therefore no
 * current chunk.
 *
 * Note that the `_alloc' parameter cannot be enclosed in ()'s because we must
 * allow `_alloc' to name a macro!
 *
 * Note that the encoded string length *includes* the terminating NULL!
 */

#define flick_cdr_decode_stringlen(_data, _link) {		\
	flick_##_link##_decode_new_glob(7);			\
	flick_##_link##_decode_new_chunk_align(4, 2, 0, 0);	\
	flick_cdr_decode_unsigned32(0, _data);			\
	flick_##_link##_decode_end_chunk(4);			\
	flick_##_link##_decode_end_glob(7);			\
}

#define flick_cdr_decode_alloced_string(_data, _link, _size) {	   \
	int _temp_len, _temp_iter;					   \
									   \
	flick_cdr_decode_stringlen(_temp_len, _link);		   \
	flick_##_link##_decode_new_glob(_temp_len);			   \
	flick_##_link##_decode_new_chunk(_temp_len);			   \
	for (_temp_iter = 0; _temp_iter < _temp_len; ++_temp_iter)	   \
		*(((char *) (_data)) + _temp_iter)			   \
			= *(((char *) _buf_current) + _temp_iter); \
	flick_##_link##_decode_end_chunk(_temp_len);			   \
	flick_##_link##_decode_end_glob(_temp_len);			   \
}

/* This is only used for 'in' strings.  Since CDR encodes the final zero,
 * we can return a pointer directly into the buffer.
 */
#define flick_cdr_decode_auto_string(_data, _alloc, _link) {	\
	int _temp_len;						\
	flick_cdr_decode_stringlen(_temp_len, _link);	\
	flick_##_link##_decode_new_glob(_temp_len);		\
	flick_##_link##_decode_new_chunk(_temp_len);		\
	(_data) = (char *)(_buf_current);		\
	flick_##_link##_decode_end_chunk(_temp_len);		\
	flick_##_link##_decode_end_glob(_temp_len);		\
}

#define flick_cdr_decode_longstring(_data, _alloc, _link) {		   \
	int _temp_len, _temp_iter;					   \
									   \
	flick_cdr_decode_stringlen(_temp_len, _link);		   \
	flick_##_link##_decode_new_glob(_temp_len);			   \
	flick_##_link##_decode_new_chunk(_temp_len);			   \
	(_data) = _alloc(_temp_len);					   \
	for (_temp_iter = 0; _temp_iter < _temp_len; ++_temp_iter)	   \
		*(((char *) (_data)) + _temp_iter)			   \
			= *(((char *) _buf_current) + _temp_iter); \
	flick_##_link##_decode_end_chunk(_temp_len);			   \
	flick_##_link##_decode_end_glob(_temp_len);			   \
}

#define flick_cdr_decode_string(_data, _alloc, _link, _size)		\
	flick_cdr_decode_longstring(_data, _alloc, _link)

#define flick_cdr_swap_decode_stringlen(_data, _link) {		\
	flick_##_link##_decode_new_glob(7);			\
	flick_##_link##_decode_new_chunk_align(4, 2, 0, 0);	\
	flick_cdr_swap_decode_unsigned32(0, _data);		\
	flick_##_link##_decode_end_chunk(4);			\
	flick_##_link##_decode_end_glob(7);			\
}

#define flick_cdr_swap_decode_auto_string(_data, _alloc, _link) {	\
	int _temp_len;							\
	flick_cdr_swap_decode_stringlen(_temp_len, _link);		\
	flick_##_link##_decode_new_glob(_temp_len);			\
	flick_##_link##_decode_new_chunk(_temp_len);			\
	(_data) = (char *)(_buf_current);			\
	flick_##_link##_decode_end_chunk(_temp_len);			\
	flick_##_link##_decode_end_glob(_temp_len);			\
}

#define flick_cdr_swap_decode_alloced_string(_data, _link, _size) {	   \
	int _temp_len, _temp_iter;					   \
									   \
	flick_cdr_swap_decode_stringlen(_temp_len, _link);		   \
	flick_##_link##_decode_new_glob(_temp_len);			   \
	flick_##_link##_decode_new_chunk(_temp_len);			   \
	for (_temp_iter = 0; _temp_iter < _temp_len; ++_temp_iter)	   \
		*(((char *) (_data)) + _temp_iter)			   \
			= *(((char *) _buf_current) + _temp_iter); \
	flick_##_link##_decode_end_chunk(_temp_len);			   \
	flick_##_link##_decode_end_glob(_temp_len);			   \
}

#define flick_cdr_swap_decode_longstring(_data, _alloc, _link) {	   \
	int _temp_len, _temp_iter;					   \
									   \
	flick_cdr_swap_decode_stringlen(_temp_len, _link);		   \
	flick_##_link##_decode_new_glob(_temp_len);			   \
	flick_##_link##_decode_new_chunk(_temp_len);			   \
	(_data) = _alloc(_temp_len);					   \
	for (_temp_iter = 0; _temp_iter < _temp_len; ++_temp_iter)	   \
		*(((char *) (_data)) + _temp_iter)			   \
			= *(((char *) _buf_current) + _temp_iter); \
	flick_##_link##_decode_end_chunk(_temp_len);			   \
	flick_##_link##_decode_end_glob(_temp_len);			   \
}

#define flick_cdr_swap_decode_string(_data, _alloc, _link, _size)	\
	flick_cdr_swap_decode_longstring((_data), _alloc, _link)
		
#ifndef TRAPEZE
#define flick_cdr_encode_IOR(_link, _data, _ref_adj)	\
	flick_cdr_encode_IOR_internal(_stream,	\
				      (_data),		\
                                      (#_link),		\
				      (_ref_adj))

#define flick_cdr_decode_IOR(_link, _data, _ref_adj)			\
	(_data) = flick_cdr_decode_IOR_internal(_stream,		\
						0 /* no swap */,	\
						(#_link),		\
						(_ref_adj))

#define flick_cdr_swap_decode_IOR(_link, _data, _ref_adj)	\
	(_data) = flick_cdr_decode_IOR_internal(_stream,	\
						1 /* swap */,	\
						(#_link),	\
						(_ref_adj))
		
#endif /* ifndef TRAPEZE */
/* End decoding section */

#endif /* __flick_encode_cdr_h */

/* End of file. */

