/*
 * Copyright (c) 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation.
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

#ifndef __flick_link_iiop_h
#define __flick_link_iiop_h

#include <flick/link/all.h>

extern int flick_seqnum;

typedef struct FLICK_BUFFER {
	void *buf_start;    /* start of current message */
	void *buf_current;   /* position in current message */
	void *buf_end;       /* end of current message */

	/* These not used for encoding and sending !!! */
	void *real_buf_start; /* start of allocated block w/multiple messages*/
	void *buf_read;       /* unfilled part of allocated block */
	void *real_buf_end;   /* end of allocated block */
} *flick_marshal_stream_t, FLICK_BUFFER;

#define _buf_start (_stream->buf_start)
#define _buf_current (_stream->buf_current)
#define _buf_end (_stream->buf_end)

/* Used for both client request & server reply... */
#define flick_iiop_build_message_header(msg_type) {			\
	((char *)_buf_start)[0] = 'G';				\
	((char *)_buf_start)[1] = 'I';				\
	((char *)_buf_start)[2] = 'O';				\
	((char *)_buf_start)[3] = 'P';				\
	((char *)_buf_start)[4] = 1;				\
	((char *)_buf_start)[5] = 0;				\
	/* little endian flag... */					\
	((short *)_buf_start)[3] = 1;				\
	((char *)_buf_start)[7] = msg_type;			\
	/* ((int *)_buf_start)[2] = size of rest of message */	\
}

#ifdef VERIFY
/* Used for both client reply & server request... */
#define flick_iiop_verify_message_header(msg_type) {		\
	if (((char *)_buf_start)[0] != 'G' ||		\
	    ((char *)_buf_start)[1] != 'I' ||		\
	    ((char *)_buf_start)[2] != 'O' ||		\
	    ((char *)_buf_start)[3] != 'P' ||		\
	    ((char *)_buf_start)[4] != 1 ||		\
	    ((char *)_buf_start)[5] != 0 ||		\
	    ((char *)_buf_start)[7] != msg_type)	\
		fprintf(stderr, "Malformed Message Header!\n");	\
}
#else
#define flick_iiop_verify_message_header(msg_type) {}
#endif

#define flick_iiop_discard_service_context_list() {			      \
	char *_buf_cursor = _buf_current;			      \
									      \
	unsigned int _service_context_seq_len;				      \
	unsigned int _service_context_len;				      \
	unsigned int _i;						      \
									      \
	_service_context_seq_len = *(unsigned int *) _buf_cursor;	      \
	_buf_cursor += 4;						      \
									      \
	if (_service_context_seq_len == 0) {				      \
		/* Zero is zero even when it's byteswapped. */		      \
	} else {							      \
		int _swap = flick_cdr_swap();				      \
									      \
		if (_swap)						      \
			_service_context_seq_len			      \
				= swap_unsigned32(_service_context_seq_len);  \
									      \
		for (_i = 0; _i < _service_context_seq_len; ++_i) {	      \
			/* Skip the `context_id', an unsigned long. */	      \
			_buf_cursor += 4;				      \
			_service_context_len				      \
				= *(unsigned int *) _buf_cursor;	      \
			if (_swap)					      \
				_service_context_len			      \
					= swap_unsigned32(		      \
						_service_context_len);	      \
			_buf_cursor += 4;				      \
			/*						      \
			 * Note: It is always correct to 4-byte align here,   \
			 * because every service context is followed by an    \
			 * unsigned long.  Even the last is followed by an    \
			 * unsigned long in the GIOP request or reply header. \
			 */						      \
			_buf_cursor += (_service_context_len + 3) & ~3;	      \
		}							      \
	}								      \
									      \
	_buf_current = _buf_cursor;				      \
}

#define flick_iiop_discard_request_id() {	\
	((char *) _buf_current) += 4;	\
}

/* Initialization / Exit code */
#define flick_iiop_client_start_encode() {				     \
	  /* The buffers were allocated when the BOA was created.	     \
	     The connection to the remote host will be made in		     \
	     flick_client_send_request(). */				     \
									     \
	_stream = _obj->boa->in;					     \
	/* Build the `GIOP::MessageHeader' (note Request == 0). */	     \
	flick_iiop_build_message_header(0);				     \
	/* The `GIOP::RequestHeader.service_context', a sequence. */	     \
	/* For now, we always encode an empty sequence. */		     \
	((int *)_buf_start)[3] = 0;					     \
	/* The `GIOP::RequestHeader.request_id'. */			     \
	((int *)_buf_start)[4] = ++flick_seqnum;			     \
	/* The `GIOP::RequestHeader.response_expected' flag is set later. */ \
	/*								     \
	 * From here, we need to put the object key, operation string,	     \
	 * plus the 'Principal' opaque interface.			     \
	 */								     \
	_buf_current = &(((int *)_buf_start)[6]);			     \
}

#define flick_iiop_client_end_encode() /* Nothing to do. */

#define flick_iiop_client_set_response_expected(val) {			\
	/*								\
	 * 20 ==  12 `GIOP::MessageHeader' bytes			\
	 *       + 4 `GIOP::RequestHeader.service_context' bytes	\
	 *       + 4 `GIOP::RequestHeader.requist_id' bytes		\
	 *								\
	 * XXX --- This assumes that the service context list is a	\
	 * zero-element sequence.					\
	 */								\
	((char *) _buf_start)[20] = (val);			\
}

#define flick_iiop_client_start_decode() {				\
	_stream = _obj->boa->out;					\
	/* Verify the reply GIOP MessageHeader (note Reply == 1). */	\
	flick_iiop_verify_message_header(1);				\
	/* Move the stream pointer to after the GIOP MessageHeader. */	\
	_buf_current = ((char *) _buf_start) + 12;	\
	flick_iiop_discard_service_context_list();			\
	flick_iiop_discard_request_id();				\
}

#define flick_iiop_client_end_decode() {}

#define flick_iiop_server_start_decode() {				 \
	_stream = InHeadP;						 \
	/* Verify the request GIOP MessageHeader (note Request == 0). */ \
	flick_iiop_verify_message_header(0);				 \
	/*								 \
	 * NOTE: The server dispatch loop sets the stream pointer to	 \
	 * point to the start of the operation name --- i.e., we've	 \
	 * already parsed up to the `operation' field of the GIOP	 \
	 * RequestHeader.  See the function `find_implementation'.	 \
	 */								 \
}

#define flick_iiop_server_end_decode() /* Nothing to do. */

#define flick_iiop_server_start_encode() {				     \
	_stream = OutHeadP;						     \
	flick_iiop_build_message_header(1);	/* Reply = 1 */		     \
	((unsigned int *)_buf_start)[3] = 0;	/* ServiceContext */ \
	((unsigned int *)_buf_start)[4] = _request_id;		     \
	_buf_current = &(((int *)_buf_start)[5]);	     \
}

#define flick_iiop_server_restart_encode()		\
	_buf_current = &(((int *)_buf_start)[5]);	

#define flick_iiop_server_end_encode() {}

/* Globbing & Chunking code */
/*
 * XXX --- The IIOP runtime function `flick_cdr_encode_IOR_internal' knows how
 * globbing is implemented.  If you change this function you must also change
 * the globbing in that runtime function.
 */
#define flick_iiop_encode_new_glob(max_size) {				  \
	if (_buf_end - _buf_current < (int) (max_size)) { \
		int ofs = _buf_current - _buf_start;	  \
		int siz = ofs + max_size;				  \
									  \
		_buf_start = (void *) realloc(_buf_start, \
						      siz);		  \
		_buf_current = _buf_start + ofs;	  \
		_buf_end = _buf_start + siz;		  \
	}								  \
}

#define flick_iiop_encode_end_glob(max_size)	/* We don't do squat */
#define flick_iiop_encode_new_chunk(size)	/* Don't do anything */
#define flick_iiop_encode_end_chunk(size)	\
	_buf_current += size;
#define flick_iiop_encode_new_chunk_align(size, final_bits, init_bits, init_ofs) {	\
	unsigned int _align = (1 << (final_bits)) - 1;					\
	_buf_current =								\
		(void *)(((unsigned int)(_buf_current + _align)) & ~_align);	\
}

#define flick_iiop_encode_new_glob_plain(max_size) {			\
  if ( _buf_end - _buf_current  < max_size) {		\
    int ofs = _buf_current - _buf_start;		\
    int siz = ofs + max_size;						\
    _buf_start = (void *)realloc(_buf_start, siz);	\
    _buf_current = _buf_start + ofs;			\
    _buf_end = _buf_start + siz;			\
  }									\
}
#define flick_iiop_encode_end_glob_plain(max_size)	/* We don't do squat */
#define flick_iiop_encode_new_chunk_plain(size)	/* Don't do anything */
#define flick_iiop_encode_end_chunk_plain(size)	\
	_buf_current += size;

#define flick_iiop_decode_new_glob(max_size)	/* We don't do squat */
#define flick_iiop_decode_end_glob(max_size)	/* Again - do nothing */
#define flick_iiop_decode_new_chunk(size)	/* Fait rien */
#define flick_iiop_decode_end_chunk(size)	\
	_buf_current += size;
#define flick_iiop_decode_new_chunk_align(size, final_bits, init_bits, init_ofs) {	\
	unsigned int _align = (1 << (final_bits)) - 1;					\
	_buf_current =								\
		(void *)(((unsigned int)(_buf_current + _align)) & ~_align);	\
}

/* The following stuff probably belongs in corba_on_iiop.h */

#define flick_iiop_client_encode_target(ref, _ofs, ENCNAME) {						\
	unsigned int __len = (ref)->key._length;							\
	flick_iiop_encode_new_chunk(4);									\
	flick_iiop_encode_new_glob(7 + __len);								\
	flick_##ENCNAME##_encode_unsigned32(0, __len);							\
	flick_iiop_encode_end_chunk(4);									\
	memcpy(_buf_current, (ref)->key._buffer, __len);						\
	_buf_current = (void *)(((unsigned int)(_buf_current + __len + 3)) & ~3);		\
	flick_iiop_encode_end_glob(7 + __len);								\
}



#define flick_iiop_client_decode_target(ref, _ofs, ENCNAME)
#define flick_iiop_server_encode_target(ref, _ofs, ENCNAME)
#define flick_iiop_server_decode_target(ref, _ofs, ENCNAME)	\
	(ref) = _this_obj;

#define flick_iiop_mark_port_for_cleanup(a, b)

typedef struct flick_client_struct *FLICK_CLIENT;

#define _typedef___FLICK_SERVER
typedef flick_operation_success_t (*FLICK_SERVER)(flick_marshal_stream_t, flick_marshal_stream_t, unsigned long, FLICK_CLIENT);
typedef int mom_server_t(flick_marshal_stream_t, flick_marshal_stream_t);

#include <flick/link/ORB.h>
#include <flick/pres/corba.h>

typedef struct flick_client_struct {
	CORBA_BOA boa;
	CORBA_ReferenceData key;
	const char *type_id;		/* considered a const literal string */
	unsigned int type_id_len;	/* strlen(typeid) */
	char *ior;			/* object's cached IOR */
	unsigned int ior_len;
	
	/* server side only */
	FLICK_SERVER server_func; 
} FLICK_CLIENT_STRUCT;

/* types for various presentations */
typedef FLICK_CLIENT mom_ref_t;
typedef flick_operation_success_t flick_dispatch_t(FLICK_BUFFER *, FLICK_BUFFER *, unsigned long, FLICK_CLIENT);


/* library routines */
int flick_client_send_request(FLICK_CLIENT cli,
			      FLICK_BUFFER *buf);
int flick_client_get_reply(FLICK_CLIENT cli,
			   FLICK_BUFFER *buf);
int flick_server_get_request(int socket,
			     FLICK_BUFFER *buf);
int flick_server_send_reply(int socket,
			    FLICK_BUFFER *buf);

char *find_system_exception_id(char *name, int size);
	
int flick_server_send_exception(int socket,
				unsigned long request_id,
				const char *exception_type);

#define flick_buffer_contains_more(buf)		\
	((buf)->buf_end < (buf)->buf_read)


/*
 * Runtime marshal/unmarshal functions.
 */
void         flick_cdr_encode_IOR_internal(flick_marshal_stream_t stream,
					   FLICK_CLIENT obj,
					   const char *link,
					   int ref_adjust);
FLICK_CLIENT flick_cdr_decode_IOR_internal(flick_marshal_stream_t stream,
					   int cdr_swap,
					   const char *link,
					   int ref_adjust);

#endif /* __flick_link_iiop_h */

/* End of file. */

