/*
 * Copyright (c) 1996, 1997 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation, and (3) all advertising
 * materials mentioning features or use of this software display the following
 * acknowledgement: ``This product includes software developed by the
 * Computer Systems Laboratory at the University of Utah.''
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

/* 
 * Modified by Uwe Dannowski, Dresden University of Technology
 */

#ifndef __flick_link_l4_h
#define __flick_link_l4_h

#define VERSION_X_OPTIMIZATION

#include <flick/link/all.h>
#include <l4/ipc.h>


// currently 4K for direct strings
#ifndef L4_MAX_RPC_BUFFER_SIZE
#define L4_MAX_RPC_BUFFER_SIZE  128
#endif

#ifndef L4_MAX_RPC_STRING_DOPES
#define L4_MAX_RPC_STRING_DOPES 2
#endif

// we define some unix-error codes here (if they are not already defined...)
#ifndef EINVAL
#define EIO 5
#define EINVAL 22
#endif

#define DISPATCH_ACK_SEND  1
#define DISPATCH_NO_REPLY  2

#define FLICK_L4_PROPAGATION  1


typedef struct {
    l4_fpage_t fpage;
    l4_msgdope_t size_dope;
    l4_msgdope_t snd_dope;
    dword_t buffer[L4_MAX_RPC_BUFFER_SIZE];
    l4_strdope_t str_dope[L4_MAX_RPC_STRING_DOPES];

    /* this is status for msg-composing */
    byte_t nr_rcv_strings;
    byte_t nr_snd_strings;
    byte_t nr_fpages;
    void* sndmsg;
} l4_ipc_buffer_t;


typedef l4_threadid_t SmService;

typedef struct {
    l4_threadid_t client_tid;
    l4_threadid_t propdest_tid;
    l4_ipc_buffer_t *msgbuf;
    l4_ipc_buffer_t *prop_msgbuf;
    l4_msgdope_t ipc_status;
    l4_timeout_t timeout;
} SmRequest;


L4_INLINE void
set_receive_fpage(SmRequest *request, l4_fpage_t fpage)
{
    request->msgbuf->fpage = fpage;
}

L4_INLINE void
set_server_rcvstring(SmRequest *request, dword_t size, dword_t addr)
{
    request->msgbuf->str_dope[request->msgbuf->nr_rcv_strings].rcv_size = size;
    request->msgbuf->str_dope[request->msgbuf->nr_rcv_strings].rcv_str = addr;
    request->msgbuf->nr_rcv_strings++;
}

L4_INLINE void
flick_init_request(SmRequest *request, l4_ipc_buffer_t *buffer)
{
    request->msgbuf = buffer;
    request->msgbuf->fpage.fpage = 0;
    request->msgbuf->nr_snd_strings = 0;
    request->msgbuf->nr_rcv_strings = 0;
    request->msgbuf->nr_fpages = 0;
    request->timeout = L4_IPC_NEVER;
}

#define snd_refstring(size, addr) (l4_strdope_t){size, addr, 0, 0}
#define rcv_refstring(size, addr) (l4_strdope_t){0, 0, size, addr}
#define sndrcv_refstring(sndsize, sndaddr, rcvsize, rcvaddr) \
        (l4_strdope_t){sndsize, sndaddr, rcvsize, rcvaddr}


/* ------------------------- */


/* Globbing & Chunking code */
#define flick_l4_encode_new_glob(max_size)			\
        if ((char *)_buf_current + (max_size) >			\
	    (char *)_buf_start + L4_MAX_RPC_BUFFER_SIZE)	\
		flick_error("Control buffer overflow!\n", "encode", "newglob");

#define flick_l4_encode_end_glob(max_size)/* We don't do squat */
#define flick_l4_encode_new_chunk(size)	/* Don't do anything */
#define flick_l4_encode_end_chunk(size)	\
	((char *)_buf_current) += (size);
#define flick_l4_encode_new_chunk_align(size, final_bits,		\
                                        init_bits, init_ofs) {	        \
	unsigned int _align = (1 << (final_bits)) - 1;			\
	_buf_current =							\
		(void *)(((unsigned int)((char *)_buf_current + _align)) \
			 & ~_align);					\
}

#define flick_l4_encode_new_glob_plain(max_size)	\
        flick_l4_encode_new_glob(max_size)
#define flick_l4_encode_end_glob_plain(max_size)	\
	flick_l4_encode_end_glob(max_size)
#define flick_l4_encode_new_chunk_plain(size)	\
	flick_l4_encode_new_chunk(size)
#define flick_l4_encode_end_chunk_plain(size)	\
	flick_l4_encode_end_chunk(size)

#define flick_l4_decode_new_glob(max_size) /* We don't do squat */
#define flick_l4_decode_end_glob(max_size)	/* Again - do nothing */
#define flick_l4_decode_new_chunk(size)	/* Fait rien */
#define flick_l4_decode_end_chunk(size)	\
	_buf_current += size;
#define flick_l4_decode_new_chunk_align(size, final_bits,		\
                                             init_bits, init_ofs) {	\
	unsigned int _align = (1 << (final_bits)) - 1;			\
	_buf_current =							\
		(void *)(((unsigned int)((char *)_buf_current + _align))\
			  & ~_align);					\
}

/* End memory stuff */

/**********************************************************************/
/*                                                                    */
/*                   Normal L4-Version                                */
/*                                                                    */
/**********************************************************************/


#ifndef __L4_VERSION_X__

#error "Flick-Bindings are not made for normal L4 - use Version X !!!"

/*----------------------------------------------------------------------*/
/*                the client-specific encoding                          */
/*----------------------------------------------------------------------*/
#define flick_l4_client_start_encode()                                 \
  _buf_current = _buf_start = (char*)desc->msgbuf->buffer;            



#define flick_l4_client_end_encode()                                    \
{	                                                                \
  int nr_dw = (_buf_current - _buf_start + 3) / 4;                      \
  if (nr_dw > 2) {						        \
      server_id->msgbuf->snd_dope = L4_IPC_DOPE(nr_dw, 0);              \
      server_id->sndmsg = (void*)server_id->msgbuf;                     \
  }                                                                     \
  else                                                                  \
    server_id->sndmsg = 0;                                              \
  server_id->msgbuf->fpage = server_id->rcv_fpage;                      \
  server_id->msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, 0);\
}

// this function takes the "optimized" fpage answer in mind. 
// only if an fpage is expected we can drop the exception code at the server
// and reconstruct it at the client side 
#define flick_l4_client_start_decode()                                 \
{                                                                      \
  if (!(server_id->rcv_fpage.fpage))                                   \
      _buf_current = _buf_start = (char*)server_id->msgbuf->buffer;    \
  else {                                                               \
      _buf_current = _buf_start = (char*)server_id->msgbuf->buffer[4]; \
      if (_rc.md.dwords <= 2) {                                        \
	  (char*)server_id->msgbuf->buffer[4] = exc_l4_no_exception;   \
      }                                                                \
  }                                                                    \
}

// for safety reasons we set the recv-fpage to 0
#define flick_l4_client_end_decode()                                  \
  server_id->rcv_fpage.fpage = 0;



/*----------------------------------------------------------------------*/
/*                the server-specific encoding                          */
/*----------------------------------------------------------------------*/

// that includes the flexpage encoding
// fpages are sent as the first 2 dwords and than we have two
// additional separator dwords

#define flick_l4_server_start_encode()                                   \
    _buf_current = _buf_start = (char*)desc->msgbuf->buffer;


#define flick_l4_server_end_encode()                                     \
{	                                                                 \
  int nr_dw = (_buf_current - _buf_start + 3) / 4;                       \
  if (!(client_id->snd_fpage.fpage))                                     \
  {                                                                      \
      if (nr_dw > 2) {						         \
          client_id->msgbuf->snd_dope = L4_IPC_DOPE(nr_dw, 0);           \
          client_id->sndmsg = (void*)client_id->msgbuf;                  \
      }                                                                  \
      else                                                               \
          client_id->sndmsg = 0;                                         \
  }                                                                      \
  else                                                                   \
  {                                                                      \
      /* here we can add a "special special" case */                     \
      /* without tranfering anything except the fpage and */             \
      /* emulating the "no exception" on the other side */               \
     if ((nr_dw == 1) /* && (_ev->_type == exc_l4_no_exception) */ )     \
     {                                                                   \
          /* we don't transfer the "separator" fpage (0,0) */            \
	  client_id->sndmsg = L4_IPC_SHORT_FPAGE;                        \
      }                                                                  \
      else {                                                             \
	  client_id->msgbuf->snd_dope = L4_IPC_DOPE(nr_dw + 4, 0);       \
          client_id->sndmsg = (void*)((dword_t)client_id->msgbuf | 2);   \
     }                                                                   \
  }                                                                      \
  client_id->msgbuf->fpage.fpage = 0;                                    \
  client_id->snd_fpage.fpage = 0;                                        \
  client_id->msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, 0); \
}

#define flick_l4_server_start_decode()                                   \
  _buf_current = _buf_start = (char*)client_id->msgbuf->buffer;          


#define flick_l4_server_end_decode()                                     \
  client_id->msgbuf->fpage.fpage = 0;                                    

#define flick_l4_return_if_propagating()                                 \
  if (is_propagating(desc)) return FLICK_OPERATION_PROPAGATING


/*----------------------------------------------------------------------*/
/*                the send-functions                                    */
/*----------------------------------------------------------------------*/

L4_INLINE l4_msgdope_t
flick_client_send_request_get_reply(l4_rpc_server_t* server)
{
    l4_msgdope_t result;    
    l4_i386_ipc_call(server->server_tid,
		     server->sndmsg, 
		     server->msgbuf->buffer[0], server->msgbuf->buffer[1],
		     (void *)server->msgbuf, 
		     &server->msgbuf->buffer[0], &server->msgbuf->buffer[1],
		     L4_IPC_NEVER, &result);
    return result;
}

L4_INLINE l4_msgdope_t
flick_server_get_request(l4_rpc_client_t* client)
{
    l4_msgdope_t result;   
    client->msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, 0); 
    l4_i386_ipc_wait(&client->client_tid,
		     (void *)client->msgbuf, 
		     &client->msgbuf->buffer[0], &client->msgbuf->buffer[1],
		     L4_IPC_NEVER, &result);
    return result;
}

L4_INLINE l4_msgdope_t
flick_server_send_reply(l4_rpc_client_t* client)
{
    l4_msgdope_t result;
    l4_i386_ipc_send(client->client_tid,
		     client->sndmsg, 
		     client->msgbuf->buffer[0], client->msgbuf->buffer[1],
		     L4_IPC_NEVER, &result);
    return result;
}

#if 1
L4_INLINE l4_msgdope_t
flick_server_reply_and_wait(l4_rpc_client_t* client)
{
    l4_msgdope_t result;
    client->msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, 0);
    l4_i386_ipc_reply_and_wait(client->client_tid,
			       client->sndmsg, 
			       client->msgbuf->buffer[0], 
			       client->msgbuf->buffer[1],
			       &client->client_tid,
			       (void *)client->msgbuf, 
			       &client->msgbuf->buffer[0], 
			       &client->msgbuf->buffer[1],
			       L4_IPC_NEVER, &result);
    return result;
}

#else

L4_INLINE l4_msgdope_t
flick_server_reply_and_wait(l4_rpc_client_t* client)
{
    l4_msgdope_t result;
    client->msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, 0);
    if (l4_i386_ipc_send(client->client_tid,
			 client->sndmsg, 
			 client->msgbuf->buffer[0], 
			 client->msgbuf->buffer[1],
			 L4_IPC_NEVER, &result))
	return result;
    
    l4_i386_ipc_wait(&client->client_tid,
		     (void*)client->msgbuf, 
		     &client->msgbuf->buffer[0], &client->msgbuf->buffer[1],
		     L4_IPC_NEVER, &result);
    return result;
}

#endif

#else __L4_VERSION_X__

/**********************************************************************/
/*                                                                    */
/*                   L4-Version X                                     */
/*                                                                    */
/**********************************************************************/

/*----------------------------------------------------------------------*/
/*                the client-specific encoding                          */
/*----------------------------------------------------------------------*/
#define flick_l4_client_start_encode()                                  \
  _buf_current = _buf_start = (char*)_msgbuf->buffer;                   \
  _msgbuf->nr_fpages = 0;                                               \
  _msgbuf->nr_snd_strings = 0;                                          \
  _msgbuf->nr_rcv_strings = 0;                                          \
  _msgbuf->fpage.fpage = 0;


#define flick_l4_client_end_encode()                                    \
{	                                                                \
  int nr_dw = (_buf_current - _buf_start + 3) / 4;                      \
  if ((nr_dw > 3) || _msgbuf->nr_snd_strings) {                         \
      _msgbuf->snd_dope = L4_IPC_DOPE(nr_dw, _msgbuf->nr_snd_strings);  \
      _msgbuf->sndmsg = (void*)_msgbuf;                                 \
  }                                                                     \
  else                                                                  \
      _msgbuf->sndmsg = 0;                                              \
  _msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, _msgbuf->nr_rcv_strings); \
}

#define flick_l4_client_start_decode()                                 \
{                                                                      \
  _buf_current = _buf_start = (char*)_msgbuf->buffer;                  \
}

#define flick_l4_client_end_decode()                                  

#define flick_l4_client_set_rcvfpage()                                 \
  _msgbuf->fpage = rcvfpage;

/*----------------------------------------------------------------------*/
/*                the propagation-specific encoding                     */
/*----------------------------------------------------------------------*/
#define flick_l4_propagation_start_encode()                             \
        _msgbuf = request->prop_msgbuf;                                 \
        flick_l4_client_start_encode();                                 \
	request->propdest_tid = service;                                \
	_ev->_type = exc_l4_propagation;                                




#define flick_l4_propagation_end_encode()                               \
{	                                                                \
  int nr_dw = (_buf_current - _buf_start + 3) / 4;                      \
  if ((nr_dw > 3) || _msgbuf->nr_snd_strings) {                         \
      _msgbuf->snd_dope = L4_IPC_DOPE(nr_dw, _msgbuf->nr_snd_strings);  \
      _msgbuf->sndmsg = (void*)_msgbuf;                                 \
  }                                                                     \
  else                                                                  \
      _msgbuf->sndmsg = 0;                                              \
}

/* propagation never decodes */


/*----------------------------------------------------------------------*/
/*                the server-specific encoding                          */
/*----------------------------------------------------------------------*/


#define flick_l4_server_start_encode()                                   \
{                                                                        \
  _msgbuf = request->msgbuf;                                             \
  _buf_current = _buf_start = (char*)_msgbuf->buffer;                    \
  _msgbuf->nr_fpages = 0;                                                \
  _msgbuf->nr_snd_strings = 0;                                           \
}

#define flick_l4_server_end_encode()                                     \
{	                                                                 \
  int nr_dw = (_buf_current - _buf_start + 3) / 4;                       \
  if ((_msgbuf->nr_fpages==1 && (nr_dw == 5)) || (nr_dw <= 3))           \
      _msgbuf->sndmsg = 0;                                               \
  else {                                                                 \
      _msgbuf->snd_dope = L4_IPC_DOPE(nr_dw, _msgbuf->nr_snd_strings);   \
      _msgbuf->sndmsg = (void*)_msgbuf;                                  \
  }                                                                      \
  if (_msgbuf->nr_fpages)                                                \
      _msgbuf->sndmsg = (void*)((dword_t)_msgbuf->sndmsg | 2);           \
  _msgbuf->size_dope = L4_IPC_DOPE(L4_MAX_RPC_BUFFER_SIZE, _msgbuf->nr_rcv_strings);      \
}

#define flick_l4_server_start_decode()                                   \
  _buf_current = _buf_start = (char*)_msgbuf->buffer;          

#define flick_l4_server_end_decode()                                     


#ifndef VERSION_X_OPTIMIZATION

L4_INLINE l4_msgdope_t
flick_client_call(SmService service, l4_ipc_buffer_t* msgbuf)
{
    l4_msgdope_t result;
    l4_i386_ipc_call((l4_threadid_t)service,
		     msgbuf->sndmsg, 
		     msgbuf->buffer[1], msgbuf->buffer[2], msgbuf->buffer[0],
		     (void *)msgbuf, 
		     &msgbuf->buffer[1],&msgbuf->buffer[2],&msgbuf->buffer[0],
		     L4_IPC_NEVER, &result);
    return result;
}

L4_INLINE l4_msgdope_t
flick_server_wait(SmRequest *request)
{
    l4_i386_ipc_wait(&request->client_tid,
		     (void *)request->msgbuf, 
		     &request->msgbuf->buffer[1], &request->msgbuf->buffer[2],
		     &request->msgbuf->buffer[0],
		     request->timeout, &request->ipc_status);
    return desc->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_send(SmRequest *request)
{
    l4_i386_ipc_send(request->tid,
		     request->msgbuf->sndmsg, 
		     request->msgbuf->buffer[1], request->msgbuf->buffer[2],
		     request->msgbuf->buffer[0],
		     request->timeout, &request->ipc_status);
    return request->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_reply_and_wait(SmRequest *request)
{
    l4_i386_ipc_reply_and_wait(request->client_tid,
			       request->msgbuf->sndmsg, 
			       request->msgbuf->buffer[1], 
			       request->msgbuf->buffer[2],
			       request->msgbuf->buffer[0],
			       &request->client_tid,
			       (void *)request->msgbuf, 
			       &request->msgbuf->buffer[1], 
			       &request->msgbuf->buffer[2],
			       &request->msgbuf->buffer[0],
			       request->timeout, &request->ipc_status);
    return request->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_propagate_and_wait(SmRequest *request)
{
    l4_i386_ipc_propagate_and_wait(request->propdest_tid,
				   request->client_tid,
				   request->prop_msgbuf->sndmsg, 
				   request->prop_msgbuf->buffer[1], 
				   request->prop_msgbuf->buffer[2],
				   request->prop_msgbuf->buffer[0],
				   &request->client_tid,
				   (void *)request->msgbuf, 
				   &request->msgbuf->buffer[1], 
				   &request->msgbuf->buffer[2],
				   &request->msgbuf->buffer[0],
				   request->timeout, &request->ipc_status);
    return request->ipc_status;
}

#else

L4_INLINE l4_msgdope_t
flick_client_call(SmService service, l4_ipc_buffer_t *msgbuf)
{
    l4_msgdope_t result;
    l4_timeout_t timeout = L4_IPC_NEVER;

   asm volatile(
      "pushl	%%ebp		\n\t"		/* save ebp, no memory 
						   references ("m") after 
						   this point */
      "movl	%%edi, %%ebp	\n\t"
      "movl     4(%%edx), %%ebx \n\t"
      "movl     8(%%edx), %%edi \n\t"
      "movl     (%%edx), %%edx  \n\t"
      "int	$0x30		\n\t"
      "popl	%%ebp		\n\t"		/* restore ebp, no memory 
						   references ("m") before 
						   this point */
      : 
      "=a" (result),    			/* EAX, 0 	*/
      "=d" (msgbuf->buffer[1]),  		/* EDX, 1 	*/
      "=b" (msgbuf->buffer[2]),		        /* EBX, 2 	*/
      "=D" (msgbuf->buffer[0])                  /* EDI, 3       */
      :
      "c" (timeout),			        /* ECX, 4 	*/
      "D" ((int)msgbuf),                        /* EDI, 5, rcv msg -> ebp */
      "S" (service),			        /* ESI, 6, dest	  */
      "0" ((int)msgbuf->sndmsg),	        /* EAX, 0  	*/
      "1" (msgbuf->buffer)	                /* EDX, 1,	*/
      :
      "esi", "ecx"
      );
   return result;
}


L4_INLINE l4_msgdope_t
flick_server_wait(SmRequest *request)
{
      asm volatile(
      "pushl	%%ebp		\n\t"		/* save ebp, no memory 
						   references ("m") after 
						   this point */
      "movl	%%ebx,%%ebp	\n\t" 
      "int	$0x30		\n\t"
      "popl	%%ebp		\n\t"		/* restore ebp, no memory 
						   references ("m") before 
						   this point */
      : 
      "=a" (request->ipc_status),			/* EAX,0 */
      "=d" (request->msgbuf->buffer[1]),		/* EDX,1 */
      "=b" (request->msgbuf->buffer[2]),		/* EBX,2 */
      "=D" (request->msgbuf->buffer[0]),		/* EDI,3 */
      "=S" (request->client_tid)        	        /* ESI,4 */
      :
      "c" (request->timeout),      		        /* ECX, 5 	*/
      "0" (L4_IPC_NIL_DESCRIPTOR),		        /* EAX, 0 	*/
      "2" ((int)request->msgbuf | L4_IPC_OPEN_IPC)	/* EBX, 2, rcv_msg -> EBP */
      :
      "ecx", "memory"
      );
    return request->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_send(SmRequest *request)
{
    asm volatile(
      "pushl	%%ebp		\n\t"		/* save ebp, no memory 
						   references ("m") after 
						   this point */
      "movl	%6 ,%%ebp	\n\t" 
      "int	$0x30		\n\t"
      "popl	%%ebp		\n\t"		/* restore ebp, no memory 
						   references ("m") before 
						   this point */
      : 
      "=a" (request->ipc_status)		/* EAX,0 	*/
      :
      "d" (request->msgbuf->buffer[1]),		/* EDX, 1	*/
      "c" (request->timeout),			/* ECX, 2	*/
      "b" (request->msgbuf->buffer[2]),		/* EBX, 3	*/
      "D" (request->msgbuf->buffer[0]),	        /* EDI, 4	*/
      "S" (request->client_tid),       	        /* ESI, 5	*/
      "i" (L4_IPC_NIL_DESCRIPTOR),		/* Int, 6 	*/
      "0" ((int)request->msgbuf->sndmsg)        /* EAX, 0 	*/
      :
      "esi", "edi", "ebx", "ecx", "edx", "memory"
      );
    return request->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_reply_and_wait(SmRequest *request)
{
    asm volatile(
      /* eax, edx, ebx loaded, 
       * edi contains rcv buffer address, must be moved to ebp,
       * esi contains address of destination id,
       * $5  address of src id
       */
      "pushl	%%ebp		\n\t"		/* save ebp, no memory 
						   references ("m") after 
						   this point */
      "movl	%%edi, %%ebp	\n\t"
      "movl     4(%%edx), %%ebx \n\t"           /* get send values */
      "movl     8(%%edx), %%edi \n\t"
      "movl     (%%edx), %%edx  \n\t"
      "int	$0x30		\n\t"
      "popl	%%ebp		\n\t"		/* restore ebp, no memory 
						   references ("m") before 
						   this point */
      : 
      "=a" (request->ipc_status),		/* EAX, 0 	*/
      "=d" (request->msgbuf->buffer[1]),	/* EDX, 1 	*/
      "=b" (request->msgbuf->buffer[2]),   	/* EBX, 2 	*/
      "=D" (request->msgbuf->buffer[0]),        /* EDI, 3       */
      "=S" (request->client_tid)                /* ESI, 4       */
      :
      "c" (request->timeout),			/* ECX, 5	*/
      "D" (((int)request->msgbuf) | L4_IPC_OPEN_IPC),/* edi, 6  -> ebp rcv_msg */
      "S" (request->client_tid),		/* ESI ,4	*/
      "0" ((int)request->msgbuf->sndmsg),       /* EAX, 0 	*/
      "1" (request->msgbuf->buffer)	        /* EDX, 1 	*/
      :
      "esi", "edi", "ecx", "memory"
      );
    return request->ipc_status;
}

L4_INLINE l4_msgdope_t
flick_server_propagate_and_wait(SmRequest *request)
{
    l4_i386_ipc_propagate_and_wait(request->propdest_tid,
				   request->client_tid,
				   request->prop_msgbuf->sndmsg, 
				   request->prop_msgbuf->buffer[1], 
				   request->prop_msgbuf->buffer[2],
				   request->prop_msgbuf->buffer[0],
				   &request->client_tid,
				   (void *)request->msgbuf, 
				   &request->msgbuf->buffer[1], 
				   &request->msgbuf->buffer[2],
				   &request->msgbuf->buffer[0],
				   request->timeout, &request->ipc_status);
    return request->ipc_status;
}


#endif VERSION_X_OPTIMIZATION

#endif __L4_VERSION_X__


#endif /* __flick_link_l4ipc_h */

/* End of file. */



