/*
 * Copyright (c) 1996, 1997, 1998 The University of Utah and
 * the Computer Systems Laboratory at the University of Utah (CSL).
 * All rights reserved.
 *
 * Permission to use, copy, modify and distribute this software is hereby
 * granted provided that (1) source code retains these copyright, permission,
 * and disclaimer notices, and (2) redistributions including binaries
 * reproduce the notices in supporting documentation.
 *
 * THE UNIVERSITY OF UTAH AND CSL ALLOW FREE USE OF THIS SOFTWARE IN ITS "AS
 * IS" CONDITION.  THE UNIVERSITY OF UTAH AND CSL DISCLAIM ANY LIABILITY OF
 * ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 *
 * CSL requests users of this software to return to csl-dist@cs.utah.edu any
 * improvements that they make and grant CSL redistribution rights.
 */

#ifndef __flick_pres_corba_h
#define __flick_pres_corba_h

#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <flick/pres/all.h>

/* This is a pseudo-CORBA allocator -- It doesn't implement freeing of ports */

#define CORBA_flick_alloc(a) (void *)malloc(a)
#define CORBA_flick_free(a) free(a)

#define CORBA_free(a) free(a)
#define CORBA_alloc(a) malloc(a)

/*
 * Here are the CORBA basic type specifiers.  See Section 14.7 of the CORBA 2.0
 * specification.
 */
typedef short CORBA_short;
typedef int CORBA_long;
typedef unsigned short CORBA_unsigned_short;
typedef unsigned int CORBA_unsigned_long;
typedef float CORBA_float;
typedef double CORBA_double;
typedef char CORBA_char;
typedef char CORBA_boolean;
typedef unsigned char CORBA_octet;
typedef unsigned int CORBA_enum;
/*
 * typedef struct CORBA_any {
 *         CORBA_TypeCode_type _type;
 *         void *_value;
 * } CORBA_any;
 */

/* FLICK_CLIENT is defined by whatever link layer header you are using. */
typedef FLICK_CLIENT CORBA_Object;

/* not implemented yet */
typedef CORBA_Object CORBA_ImplementationDef;
typedef CORBA_Object CORBA_InterfaceDef;

/*
 * This definition is requested by CORBA V2.0 Spec in section 7.2.3
 * XXX Should it be CORBA_Object_OBJECT_NIL or CORBA_ORB_OBJECT_NIL instead?
 */   
#define OBJECT_NIL 0

/*
 * Here are other typedefs that are part of a CORBA presentation.  These are
 * not basic CORBA types; they are part of the ORB/BOA/Object interface.
 */

/*
 * The `struct CORBA_ORB_type' and `struct CORBA_BOA_type' are defined in
 * <flick/link/ORB.h>.
 */
typedef struct CORBA_ORB_type *CORBA_ORB;
typedef struct CORBA_BOA_type *CORBA_BOA;

#ifndef TRAPEZE
typedef struct {
	CORBA_unsigned_long _length, _maximum;
	CORBA_octet *_buffer;
} CORBA_ReferenceData;
#endif

typedef CORBA_char *CORBA_ORBid;
typedef CORBA_char *CORBA_ORB_OAid;


/*****************************************************************************/

/* These are the CORBA_exception_type constants */
typedef CORBA_unsigned_long CORBA_exception_type;
#define CORBA_NO_EXCEPTION ((CORBA_exception_type) 0)
#define CORBA_USER_EXCEPTION ((CORBA_exception_type) 1)
#define CORBA_SYSTEM_EXCEPTION ((CORBA_exception_type) 2)

/* These are the CORBA_completion_status constants */
typedef CORBA_unsigned_long CORBA_completion_status;
#define CORBA_COMPLETED_YES ((CORBA_completion_status) 0)
#define CORBA_COMPLETED_NO ((CORBA_completion_status) 1)
#define CORBA_COMPLETED_MAYBE ((CORBA_completion_status) 2)

typedef struct flick_system_exception {
	CORBA_unsigned_long minor;
	CORBA_completion_status completed;
} flick_system_exception;

typedef struct CORBA_Environment {
	CORBA_exception_type _major;
	union {
		flick_system_exception _system_except;
		void *_user_except;
	} _value;
	char *_id;
} CORBA_Environment;

/* Should only be called from Flick generated code */
#define flick_corba_init_environment() {		\
	(_ev)->_major = CORBA_NO_EXCEPTION;		\
	(_ev)->_value._system_except.minor = 0;	\
	(_ev)->_value._system_except.completed = 0;	\
	(_ev)->_id = 0;				\
}

void *CORBA_exception_value(CORBA_Environment *ev);

#define CORBA_exception_id(_ev) (_ev)->_id

/* Frees memory taken by a potential exception */
void flick_system_exception_free(void *system_except);

#define CORBA_exception_free(_ev) {					 \
	if (((_ev)->_major == CORBA_USER_EXCEPTION)			 \
	    && ((_ev)->_value._user_except)) {				 \
		CORBA_free((_ev)->_value._user_except);			 \
		(_ev)->_value._user_except = 0;				 \
	}								 \
	(_ev)->_id = 0;							 \
	(_ev)->_major = CORBA_NO_EXCEPTION;				 \
}

/* Try to match the name to the predefined system exceptions. */
char *find_system_exception_id(char *name, int size);

/* Sets a system exception without allocating memory */
#define flick_set_exception(boa, ev, exceptname, errno, completion)	\
{									\
	(ev)->_major = CORBA_SYSTEM_EXCEPTION;				\
	(ev)->_value._system_except.minor = (errno);			\
	(ev)->_value._system_except.completed = (completion);		\
	(ev)->_id = (exceptname);					\
}

#define corba_except(name)					\
typedef flick_system_exception name;				\
name *##name##__alloc();

/*
 * Here are the standard CORBA system exception.
 *
 * If you update this list, you should also update the CORBA runtime library.
 * See `corba_excepts.c'.
 */
corba_except(CORBA_UNKNOWN)
#define ex_CORBA_UNKNOWN  "IDL:omg.org/CORBA/UNKNOWN:1.0"
corba_except(CORBA_BAD_PARAM)
#define ex_CORBA_BAD_PARAM  "IDL:omg.org/CORBA/BAD_PARAM:1.0"
corba_except(CORBA_NO_MEMORY)
#define ex_CORBA_NO_MEMORY  "IDL:omg.org/CORBA/NO_MEMORY:1.0"
corba_except(CORBA_IMP_LIMIT)
#define ex_CORBA_IMP_LIMIT  "IDL:omg.org/CORBA/IMP_LIMIT:1.0"
corba_except(CORBA_COMM_FAILURE)
#define ex_CORBA_COMM_FAILURE  "IDL:omg.org/CORBA/COMM_FAILURE:1.0"
corba_except(CORBA_INV_OBJREF)
#define ex_CORBA_INV_OBJREF  "IDL:omg.org/CORBA/INV_OBJREF:1.0"
corba_except(CORBA_NO_PERMISSION)
#define ex_CORBA_NO_PERMISSION  "IDL:omg.org/CORBA/NO_PERMISSION:1.0"
corba_except(CORBA_INTERNAL)
#define ex_CORBA_INTERNAL  "IDL:omg.org/CORBA/INTERNAL:1.0"
corba_except(CORBA_MARSHAL)
#define ex_CORBA_MARSHAL  "IDL:omg.org/CORBA/MARSHAL:1.0"
corba_except(CORBA_INITIALIZE)
#define ex_CORBA_INITIALIZE  "IDL:omg.org/CORBA/INITIALIZE:1.0"
corba_except(CORBA_NO_IMPLEMENT)
#define ex_CORBA_NO_IMPLEMENT  "IDL:omg.org/CORBA/NO_IMPLEMENT:1.0"
corba_except(CORBA_BAD_TYPECODE)
#define ex_CORBA_BAD_TYPECODE  "IDL:omg.org/CORBA/BAD_TYPECODE:1.0"
corba_except(CORBA_BAD_OPERATION)
#define ex_CORBA_BAD_OPERATION  "IDL:omg.org/CORBA/BAD_OPERATION:1.0"
corba_except(CORBA_NO_RESOURCES)
#define ex_CORBA_NO_RESOURCES  "IDL:omg.org/CORBA/NO_RESOURCES:1.0"
corba_except(CORBA_NO_RESPONSE)
#define ex_CORBA_NO_RESPONSE  "IDL:omg.org/CORBA/NO_RESPONSE:1.0"
corba_except(CORBA_PERSIST_STORE)
#define ex_CORBA_PERSIST_STORE  "IDL:omg.org/CORBA/PERSIST_STORE:1.0"
corba_except(CORBA_BAD_INV_ORDER)
#define ex_CORBA_BAD_INV_ORDER  "IDL:omg.org/CORBA/BAD_INV_ORDER:1.0"
corba_except(CORBA_TRANSIENT)
#define ex_CORBA_TRANSIENT  "IDL:omg.org/CORBA/TRANSIENT:1.0"
corba_except(CORBA_FREE_MEM)
#define ex_CORBA_FREE_MEM  "IDL:omg.org/CORBA/FREE_MEM:1.0"
corba_except(CORBA_INV_IDENT)
#define ex_CORBA_INV_IDENT  "IDL:omg.org/CORBA/INV_IDENT:1.0"
corba_except(CORBA_INV_FLAG)
#define ex_CORBA_INV_FLAG  "IDL:omg.org/CORBA/INV_FLAG:1.0"
corba_except(CORBA_INTF_REPOS)
#define ex_CORBA_INTF_REPOS  "IDL:omg.org/CORBA/INTF_REPOS:1.0"
corba_except(CORBA_BAD_CONTEXT)
#define ex_CORBA_BAD_CONTEXT  "IDL:omg.org/CORBA/BAD_CONTEXT:1.0"
corba_except(CORBA_OBJ_ADAPTER)
#define ex_CORBA_OBJ_ADAPTER  "IDL:omg.org/CORBA/OBJ_ADAPTER:1.0"
corba_except(CORBA_DATA_CONVERSION)
#define ex_CORBA_DATA_CONVERSION  "IDL:omg.org/CORBA/DATA_CONVERSION:1.0"
corba_except(CORBA_OBJECT_NOT_EXIST)
#define ex_CORBA_OBJECT_NOT_EXIST  "IDL:omg.org/CORBA/OBJECT_NOT_EXIST:1.0"

#undef corba_except

/*
 * Here are the ORB/BOA user exceptions.
 */

#ifndef ex_CORBA_ORB_InvalidName
#define ex_CORBA_ORB_InvalidName "IDL:omg.org/CORBA/ORB/InvalidName:1.0"

typedef struct CORBA_ORB_InvalidName {
	/*
	 * There is no actual exception data, but a C struct must have at least
	 * one member.  So, we have a `_dummy'.
	 */
	CORBA_long _dummy;
} CORBA_ORB_InvalidName;

CORBA_ORB_InvalidName *CORBA_ORB_InvalidName__alloc();

#endif /* ex_CORBA_ORB_InvalidName */

/* Translate Flick internal errors to CORBA exception ids. */
char *flick_error_to_CORBA_exception_id(int errval);


/*****************************************************************************/

/********
 * Here's the ORB interface that applies (non-DII/DSI/IR stuff...)
 ********/

CORBA_ORB
CORBA_ORB_init(int *argc, char **argv,
	       CORBA_ORBid orb_identifier,
	       CORBA_Environment *ev);

CORBA_BOA
CORBA_ORB_BOA_init(CORBA_ORB ths,
		   int *argc, char **argv,
		   CORBA_ORB_OAid boa_identifier,
		   CORBA_Environment *ev);

/*
 * The following typedefs are for `CORBA::ORB::list_initial_services' and
 * `CORBA::ORB::resolve_initial_references'.  See Section 5.6 of the CORBA 2.1
 * specification.
 */
#ifndef _typedef___CORBA_ORB_ObjectId
#define _typedef___CORBA_ORB_ObjectId
typedef CORBA_char *CORBA_ORB_ObjectId;
#endif /* _typedef___CORBA_ORB_ObjectId */

#ifndef _typedef___CORBA_sequence_string
#define _typedef___CORBA_sequence_string
typedef struct CORBA_sequence_string {
  CORBA_unsigned_long _maximum;
  CORBA_unsigned_long _length;
  CORBA_char **_buffer;
} CORBA_sequence_string;
#endif /* _typedef___CORBA_sequence_string */

#ifndef _typedef___CORBA_ORB_ObjectIdList
#define _typedef___CORBA_ORB_ObjectIdList
typedef struct CORBA_sequence_string CORBA_ORB_ObjectIdList;
#endif /* _typedef___CORBA_ORB_ObjectIdList */

CORBA_ORB_ObjectIdList *
CORBA_ORB_list_initial_services(CORBA_ORB obj, CORBA_Environment *ev);

CORBA_Object
CORBA_ORB_resolve_initial_references(CORBA_ORB obj,
				     CORBA_ORB_ObjectId identifier,
				     CORBA_Environment *ev);

CORBA_char *
CORBA_ORB_object_to_readable_string(CORBA_ORB orb,
				    CORBA_Object obj,
				    CORBA_Environment *ev);

CORBA_char *
CORBA_ORB_object_to_string(CORBA_ORB orb,
			   CORBA_Object obj,
			   CORBA_Environment *ev);

CORBA_Object
CORBA_ORB_string_to_object(CORBA_ORB orb,
			   CORBA_char *str,
			   CORBA_Environment *ev);

/********
 * Here's the BOA stuff that applies...
 ********/

void
CORBA_BOA_set_exception(CORBA_BOA boa, CORBA_Environment *ev,
			CORBA_exception_type major,
			CORBA_char *exceptname, void *param);

CORBA_ReferenceData *
CORBA_BOA_get_id(CORBA_BOA ths, CORBA_Object obj, CORBA_Environment *ev);

#ifdef _typedef___FLICK_SERVER
/* Functions only needed on the server side */
CORBA_Object
CORBA_BOA_create(CORBA_BOA ths,
		 CORBA_ReferenceData *obj_key,
		 const char *type_id,
		 FLICK_SERVER,
		 CORBA_Environment *ev);

void
CORBA_BOA_impl_is_ready(CORBA_BOA ths, CORBA_Environment *ev);
#endif

/*
 * I'm not doing this stuff right now:
 *
 * void CORBA_BOA_dispose(CORBA_BOA ths, CORBA_Object obj);
 */

/*
 * Here's the Object interface that applies (non-DII/DSI/IR stuff...)
 */
CORBA_boolean
CORBA_Object_is_nil(CORBA_Object ths, CORBA_Environment *ev);

CORBA_Object
CORBA_Object_duplicate(CORBA_Object ths, CORBA_Environment *ev);

void
CORBA_Object_release(CORBA_Object ths, CORBA_Environment *ev);

CORBA_unsigned_long
CORBA_Object_hash(CORBA_Object ths,
		  CORBA_unsigned_long maximum,
		  CORBA_Environment *ev);

CORBA_boolean
CORBA_Object_is_equivalent(CORBA_Object ths,
			   CORBA_Object other_object,
			   CORBA_Environment *ev);

/* XXX --- Not implemented yet.  Currently throws an exception. */
CORBA_ImplementationDef
CORBA_Object_get_implementation(CORBA_Object ths,
				CORBA_Environment *ev);

/* XXX --- Not implemented yet.  Currently throws an exception. */
CORBA_InterfaceDef
CORBA_Object_get_interface(CORBA_Object ths, CORBA_Environment *ev);

/*
 * Tests to see if an object is descended from a given type.
 * XXX --- Not implemented yet.  Currently returns false.
 */
CORBA_boolean
CORBA_Object_is_a(CORBA_Object ths,
		  CORBA_char *logical_type_id,
		  CORBA_Environment *ev);

/*
 * Returns true if the object referred to no longer exists.
 * XXX --- Currently just returns true if the reference is OBJECT_NIL.
 */
CORBA_boolean
CORBA_Object_non_existent(CORBA_Object ths, CORBA_Environment *ev);
	

/*****************************************************************************/

/*
 * The encode/decode system exceptions don't have to deal with the _major
 * field because it has already been marshaled/unmarshaled.
 */
#define flick_corba_decode_system_exception(loc, ENCNAME, LINKNAME)	\
{									\
	flick_##ENCNAME##_decode_auto_string((loc)->_id,		\
					     null_flick_free,		\
					     LINKNAME);			\
	(loc)->_id = find_system_exception_id((loc)->_id,		\
					      strlen((loc)->_id) + 1);	\
	flick_##LINKNAME##_decode_new_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
	flick_##LINKNAME##_decode_new_chunk_align(			\
		(flick_##ENCNAME##_unsigned32_size) * 2, 2, 0, 0);	\
	flick_##ENCNAME##_decode_unsigned32				\
		(0, (loc)->_value._system_except.minor);		\
	flick_##ENCNAME##_decode_unsigned32				\
		((flick_##ENCNAME##_unsigned32_size),			\
		 (loc)->_value._system_except.completed);		\
	flick_##LINKNAME##_decode_end_chunk(				\
		(flick_##ENCNAME##_unsigned32_size) * 2);		\
	flick_##LINKNAME##_decode_end_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
}

#define flick_corba_encode_exception_type(_data, ENCNAME, LINKNAME)	      \
{									      \
	flick_##LINKNAME##_encode_new_glob(flick_##ENCNAME##_signed32_size);  \
	flick_##LINKNAME##_encode_new_chunk(flick_##ENCNAME##_signed32_size); \
        flick_##ENCNAME##_encode_signed32(0, _data);			      \
	flick_##LINKNAME##_encode_end_chunk(flick_##ENCNAME##_signed32_size); \
	flick_##LINKNAME##_encode_end_glob(flick_##ENCNAME##_signed32_size);  \
}

#define flick_corba_encode_system_exception(loc, ENCNAME, LINKNAME)	\
{									\
	flick_##ENCNAME##_encode_longstring(CORBA_exception_id(loc),	\
					    null_flick_free, LINKNAME);	\
	flick_##LINKNAME##_encode_new_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
	flick_##LINKNAME##_encode_new_chunk_align(			\
		(flick_##ENCNAME##_unsigned32_size) * 2, 2, 0, 0);	\
	flick_##ENCNAME##_encode_unsigned32				\
		(0, (loc)->_value._system_except.minor);		\
	flick_##ENCNAME##_encode_unsigned32				\
		((flick_##ENCNAME##_unsigned32_size),			\
		 (loc)->_value._system_except.completed);		\
	flick_##LINKNAME##_encode_end_chunk(				\
		(flick_##ENCNAME##_unsigned32_size) * 2);		\
	flick_##LINKNAME##_encode_end_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
									\
	CORBA_exception_free(loc);					\
}


/* These are all the error macros */

#define flick_corba_server_error(retname, errval, complete,		\
			         ENCNAME, LINKNAME, START) {		\
	flick_##LINKNAME##_server_##START##_encode();			\
	flick_##LINKNAME##_encode_new_glob(				\
		flick_##ENCNAME##_unsigned32_size);			\
	flick_##LINKNAME##_encode_new_chunk(				\
		flick_##ENCNAME##_unsigned32_size);			\
	flick_##ENCNAME##_encode_unsigned32(0, CORBA_SYSTEM_EXCEPTION);	\
	flick_##LINKNAME##_encode_end_chunk(				\
		flick_##ENCNAME##_unsigned32_size);			\
	flick_##LINKNAME##_encode_end_glob(				\
		flick_##ENCNAME##_unsigned32_size);			\
	/* XXX --- We should pass in the system exception type! */	\
	flick_##ENCNAME##_encode_longstring(				\
		flick_error_to_CORBA_exception_id((errval)),		\
		null_flick_free,					\
		LINKNAME);						\
	flick_##LINKNAME##_encode_new_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
	flick_##LINKNAME##_encode_new_chunk_align(8, 2, 0, 0);		\
	flick_##ENCNAME##_encode_unsigned32(0, 0); /* minor code */	\
	flick_##ENCNAME##_encode_unsigned32(				\
		flick_##ENCNAME##_unsigned32_size, complete);		\
	flick_##LINKNAME##_encode_end_chunk(				\
		(flick_##ENCNAME##_unsigned32_size) * 2);		\
	flick_##LINKNAME##_encode_end_glob(				\
		(flick_##ENCNAME##_unsigned32_size) * 2 + 3);		\
	flick_##LINKNAME##_server_end_encode();				\
	retname;							\
}

#define flick_corba_encode_server_error(retname, errval, ENCNAME, LINKNAME)\
	flick_corba_server_error(retname, errval, CORBA_COMPLETED_YES,	\
				 ENCNAME, LINKNAME, restart)

#define flick_corba_decode_server_error(retname, errval, ENCNAME, LINKNAME) \
	flick_corba_server_error(retname, errval, CORBA_COMPLETED_NO,	    \
				 ENCNAME, LINKNAME, start)

#define flick_corba_other_server_error(retname, errval, ENCNAME, LINKNAME) \
	flick_corba_server_error(retname, errval, CORBA_COMPLETED_MAYBE,   \
				 ENCNAME, LINKNAME, start)


#define flick_corba_client_error(retname, errval, complete, ENCNAME,	 \
				 LINKNAME)				 \
{									 \
	flick_set_exception(0 /*boa*/,					 \
			    _ev,					 \
			    flick_error_to_CORBA_exception_id((errval)), \
			    0,						 \
			    (complete));				 \
	retname;							 \
}

#define	flick_corba_encode_client_error(retname, errval, ENCNAME, LINKNAME) \
	flick_corba_client_error(retname, errval, CORBA_COMPLETED_NO,	    \
				 ENCNAME, LINKNAME)


#define flick_corba_decode_client_error(retname, errval, ENCNAME, LINKNAME) \
	flick_corba_client_error(retname, errval, CORBA_COMPLETED_MAYBE,    \
				 ENCNAME, LINKNAME)


#define flick_corba_other_client_error(retname, errval, ENCNAME, LINKNAME) \
	flick_corba_client_error(retname, errval, CORBA_COMPLETED_MAYBE,   \
				 ENCNAME, LINKNAME)

#define flick_corba_encode_mu_error(retname, errval, ENCNAME, LINKNAME)	\
		flick_error("CORBA", "encode", "marshal/unmarshal")

#define flick_corba_decode_mu_error(retname, errval, ENCNAME, LINKNAME)	\
		flick_error("CORBA", "decode", "marshal/unmarshal")

#define flick_corba_other_mu_error(retname, errval, ENCNAME, LINKNAME)	\
		flick_error("CORBA", "other", "marshal/unmarshal")

#endif /* __flick_pres_corba_h */

/* End of file. */

