/*
**	Copyright (c) 1995, Sebastian Schoenberg 
**	E-mail : ss10@irz.inf.tu-dresden.de
**
**	University of Technology Dresden, Department of Computer Science
**	Institute for Operating Systems, Databases and Computer Networks
**
**	$Id: l4pal.S,v 5.6 1999/11/09 00:42:15 danielp Exp $
**
*/

#if !defined(lint)
	.data
	.asciz	"$Id: l4pal.S,v 5.6 1999/11/09 00:42:15 danielp Exp $"
#endif


#include <pal/predef.h>

#include <pal/dc21164.h>
#include <pal/macros.h>
#include <pal/regdef.h>

#include <pal/l4pal.h>

#include <pal/debug.h>

#include <pal/gasp.h>

#include <pal/platform.h>

.MACRO pf_stack
	ldiq    AT, 0x12345678
	push    p_AT
	clr     AT
.ENDM

	

#define pf_restack\
	addq    sp, 8, sp
		

#define MAXINT	32
	
#define ICSR_K_INIT \
        ((1<<(ICSR_V_SDE-16)) | \
         (2<<(ICSR_V_SPE-16)) | \
	 (1<<(ICSR_V_FPE-16)) | \
	 (1<<(ICSR_V_CRDE-16)) | \
	 (1<<(ICSR_V_HWE-16)))
	
	.text	0x0000

	.base	zero
	

	.globl	pal_ptbase
	.globl	pal_ptroots
	.globl	pal_ptmaps


	.globl	ret_itb_fill
	.globl	ret_itb_schedule
	.globl	ret_itb_rerun
	.globl	ret_dtb_fill
	.globl	ret_dtb_schedule
	.globl	ret_dtb_rerun
	

	.globl	system_cb
	.globl	interrupt_owner
	.globl	interrupt_queue
	.globl	thread_priorities
	
/*------------------------------------------------------------------------------
** FUNCTION: pal_reset
**
**      This function is called after loading the image or after calling 
**      SWPPAL original PALcode. This can be determined by looking at the 
**      instruction pointer. If the IP is greater then 0x20000000, the code
**      is running in virtual mode and we are in Kernel mode, otherwise we
**      are in PALmode. The setup procedure MUST enable executing PALcommands
**      like mtpr, mfpr etc. in Kernel Mode !
** 
** MODE: P / K
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**      K:      Sets mode to PAL
**      P:      Establishes PALcode and initializes the AXP
*/
        HDW_VECTOR (PAL_RESET_ENTRY)
palcode:	
pal_reset_entry:
	nop
        br      a3, pal_reset_cont


/*------------------------------------------------------------------------------
** FUNCTION: pal_iaccvio_entry
**	Istream access violation or sign check error
**
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_IACCVIO_ENTRY)
pal_iaccvio_entry:
	open_frame
	debug
	sys_halt	0x1111

/*------------------------------------------------------------------------------
** FUNCTION:
**	An Interrupt has occured
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_INTERRUPT_ENTRY)
pal_interrupt_entry:
	mfpr	p7, isr				// Get summary of pending interrupts.
	srl	p7, ISR_V_HLT, p1		// Isolate the halt interrupt bit.

	mfpr	p6, ipl				// Get the IPL.
	NOP					// May delete when scheduling!!
		
	mfpr	p5, intId			// Get target IPL of pending interrupt.
#ifdef SHOW_INTR
	br	p4, 1f

	.long	((0xb8000 << 5) + (0x01 << 3))
	.long	0x00000080
1:	
	ldq_p	p4, 0x0(p4)
	sll	p5, 7, p3
	addq	p4, p3, p4
	ldl_p	p3, 0(p4)
	addq	p3, 1, p3
	and	p3, 0xff, p3
	ldiq	p2, 0x1e00
	or	p2, p3, p2
	stl_p	p2, 0(p4)

#endif	
	
	nop	
	/*
	** The HALT interrupt does not have a target IPL at which it can
	** be masked.  On a HALT interrupt, INTID indicates the NEXT highest
	** priority pending interrupt.  PALcode must first check the ISR<HLT>
	** bit to determine if a HALT interrupt has occured before proceeding
	** with the processing of interrupts with maskable target IPLs.
	*/

	blbs	p1, pal_halt_handler		// If halt interrupt, go handle ...
	/*
	** Not a HALT interrupt, so service the highest priority pending
	** interrupt whose target IPL is greater than the current processor
	** IPL.
	*/
	cmple	p5, p6, p0			// Is target IPL (ipl) > current IPL (intId)?
	bne	p0, pal_passive_release		// If not, ignore the interrupt.

	br	zero, sys_interrupt_cont	// Continue handling elsewhere ...
	

/*------------------------------------------------------------------------------
** FUNCTION: pal_itb_miss_entry
**      Istream DTB miss. Fetching page tables and load an ITB cell.
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**      modifies ITB
*/
        HDW_VECTOR (PAL_ITB_MISS_ENTRY)
pal_itb_miss_entry:
	ldiq	p6, 0x3ff 
	mfpr	p7, excAddr				// Get fault address
	ldiq	p5, 0x1c0
	mov	p7, p3

	mfpr	p0, ptPageBase				// t0 = PageBase
	bic	p7, p6, p7				// 0x3ff - clear protection area in address
	bis	p7, p5, p7				// XOR inverts only X, W, P bits 
	bic	p0, 0x3f, p2				// t2 = old.frame start base
	srl	p7, p0, p1				// t1 = table_bits (p7)
	ldiq	p6, PP_EXECUTABLE|PP_INTREE		// PP_EXEC | PP_INTREE protection required
	// MUST BE QUAD-ALIGNED !
	REPEAT
		s4addq	p1, 0, p1			// t1 = t1 * 4	
		s4addq	p1, p2, p2			// t1 = t1 * 4 + t2 -- t1 * 16 + 4

		ldq_p	p1, 0(p2)			// load lower part of gpte
		ldq_p	p0, 8(p2)			// t0 = higher part of guard

		xor	p7, p1, p7			// guard bits and flip prot.
		bic	p0, 0x3f, p2			// t2 = old.frame start base	
		srl	p7, p1, p4			// shifted guard
		and	p7, p6, p1			// P - check
		bis	p7, p5, p7			// XOR inverts only X, W, P bits 
	 	bne	p1, itb_protection_miss		// P - check
		srl	p7, p0, p1			// t1 = table_bits (p7)
		CONTZ	p4
	ENDR
	br	zero, itb_guard_fault
		
/*------------------------------------------------------------------------------
** FUNCTION: pal_dtb_miss_entry
**      Dstream DTB miss. Fetching page tables and load a DTB cell.
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**      modifies DTB
*/
        HDW_VECTOR (PAL_DTB_MISS_ENTRY)
pal_dtb_miss_entry:
	mtpr	p7, tb7					// tbX is equal to ptX
	mtpr	p0, tb0
	mtpr	p1, tb1
	mtpr	p5, tb5
	mtpr	p6, tb6
	mfpr	p7, va					// Get fault address
	mtpr	p3, tb3
	ldiq	p6, 0x3ff 
	ldiq	p5, 0x1c0
	bis	p7, p7, p3
	
	mfpr	p0, ptPageBase				// t0 = PageBase
	bic	p7, p6, p7				// 0x3ff- clear protection area in address
	mtpr	p2, tb2	
	bis	p7, p5, p7				// XOR inverts only X, W, P bits 
	mtpr	p4, tb4
	bic	p0, 0x3f, p2				// t2 = old.frame start base
	srl	p7, p0, p1				// t1 = table_bits (a0)
	ldiq	p6, PP_INTREE				// PPP_INTREE protection required
	// MUST BE QUAD-ALIGNED
	
	REPEAT
		s4addq	p1, 0, p1			// t1 = t1 * 4	
		s4addq	p1, p2, p2			// t1 = t1 * 4 + t2 -- t1 * 16 + 4

		ldq_p	p1, 0(p2)			// load lower part of gpte
		ldq_p	p0, 8(p2)			// t0 = higher part of guard

		xor	p7, p1, p7			// guard bits and flip prot.
		bic	p0, 0x3f, p2			// t2 = old.frame start base	
		srl	p7, p1, p4			// shifted guard
		and	p7, p6, p1			// P - check
		bis	p7, p5, p7			// XOR inverts only X, W, P bits 
	 	bne	p1, dtb_protection_miss		// P - check
		srl	p7, p0, p1			// t1 = table_bits (p7)
		CONTZ	p4
	ENDR
	br	zero, dtb_guard_fault

	
	
		
/*------------------------------------------------------------------------------
** FUNCTION: pal_double_miss_entry
**      Dstream ldq_v instruction. Might be we're using this for RT ?
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**      modifies DTB
*/
        HDW_VECTOR (PAL_DOUBLE_MISS_ENTRY)
pal_double_miss_entry:
	mtpr	p7, tb7					// tbX is equal to ptX
	mtpr	p0, tb0
	mtpr	p1, tb1
	mtpr	p5, tb5
	mtpr	p6, tb6
	mfpr	p7, va					// Get fault address
	mtpr	p3, tb3
	ldiq	p6, 0x3ff 
	ldiq	p5, 0x1c0
	bis	p7, p7, p3
	
	mfpr	p0, ptPageBase				// t0 = PageBase
	bic	p7, p6, p7				// 0x3ff- clear protection area in address
	mtpr	p2, tb2	
	bis	p7, p5, p7				// XOR inverts only X, W, P bits 
	mtpr	p4, tb4
	bic	p0, 0x3f, p2				// t2 = old.frame start base
	srl	p7, p0, p1				// t1 = table_bits (a0)
	ldiq	p6, PP_INTREE				// PPP_INTREE protection required
	// MUST BE QUAD-ALIGNED
	
	REPEAT
		s4addq	p1, 0, p1			// t1 = t1 * 4	
		s4addq	p1, p2, p2			// t1 = t1 * 4 + t2 -- t1 * 16 + 4

		ldq_p	p1, 0(p2)			// load lower part of gpte
		ldq_p	p0, 8(p2)			// t0 = higher part of guard

		xor	p7, p1, p7			// guard bits and flip prot.
		bic	p0, 0x3f, p2			// t2 = old.frame start base	
		srl	p7, p1, p4			// shifted guard
		and	p7, p6, p1			// P - check
		bis	p7, p5, p7			// XOR inverts only X, W, P bits 
	 	bne	p1, dtb_protection_miss		// P - check
		srl	p7, p0, p1			// t1 = table_bits (p7)
		CONTZ	p4
	ENDR
	br	zero, dtb_double_guard_fault

	
/*------------------------------------------------------------------------------
** FUNCTION: pal_unaligned_entry
**	Accessing data at an unaligned address
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	NO PALshadow registers !
*/
	HDW_VECTOR (PAL_UNALIGN_ENTRY)
pal_unaligned_entry:
#if 1	
	ldiq	p0, IDT_UNALIGNED
	br	l4_do_exception
#else
	nop
	br	debug_special
#endif
	
		
/*------------------------------------------------------------------------------
** FUNCTION: pal_d_fault_entry
**	
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_D_FAULT_ENTRY)
pal_dfault_entry:
	nop
	br	pal_dfault_cont

/*	
	open_frame
	store	(5)
	ppush	(a0, 0)
	ppush	(a1, 1)
	ppush	(a2, 2)
	ppush	(AT, 3)				// AT is thrashed while return !
	ppush	(gp, 4)
	mfpr	a0, va
	mfpr	a1, excAddr
	mfpr	a2, mmStat

	kernel	(l4_dtb_access_fault)
*/


	
/*------------------------------------------------------------------------------
** FUNCTION: pal_mchk_entry
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_MCHK_ENTRY)
pal_mchk_entry:
	ldiq	sp, 0x7777
	br	zero, pal_halt_cont
	

	
		

/*------------------------------------------------------------------------------
** FUNCTION: pal_opcdec_entry
**	invalid palcall (i.e. > 128)
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_OPCDEC_ENTRY)
pal_opcdec_entry:
#if 1
	open_frame
	push	p_r0
	mfpr	r0, excAddr
	STALL
	STALL
	debug
	sys_halt	0xdec
	ldiq	p0, IDT_INSTRUCTION
	br	l4_do_exception
#else
	// seedy debugging hack not needed now.
	ldiq	t12, 0x6970
	nop
	nop
	br	debug_special
#endif	

	
/*------------------------------------------------------------------------------
** FUNCTION: pal_arith_entry
**	arithmetic fault
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_ARITH_ENTRY)
pal_arith_entry:
#if 1	
	ldiq	p0, IDT_ARITHMETIC
	br	l4_do_exception
#else
	// seedy debugging hack not needed now.
	ldiq	t12, 0x6971
	debug
#endif	
	
/*------------------------------------------------------------------------------
** FUNCTION: pal_fen_entry
**	floating point not enabled entry
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	HDW_VECTOR (PAL_FEN_ENTRY)
pal_fen_entry:	
	sys_halt	0xfe0


	
/*=============================================================================
** PRIV PALCalls = PG 1
**
** L4 Privileged PALcode entry points
**
*/
        .text   1

/*------------------------------------------------------------------------------
** FUNCTION: pal_halt_entry
**	privileged return to console
**
** MODE: K
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
        CALL_PAL_PRIV(PAL_HALT_ENTRY)
pal_halt_entry:
	STALL
	STALL
        br      zero, pal_halt_cont


/*------------------------------------------------------------------------------
** FUNCTION: pal_ret
**	Return from Kernel mode and continue at given point
**	Oposite part of kernel ()
**
** MODE: K
**
** INPUT PARAMETERS:  
**	AT  - register to continue in PALmode.
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	Enters PALcode.
*/
	CALL_PAL_PRIV (PAL_RETPAL_ENTRY)
pal_ret_entry:
#ifdef DBG_ERROR
	ldiq	p0, 0x200000
	zap	AT, 0xf0, p1
	subq	p1, p0, p1
	bge	p1, 1f

	mfpr	$0, excAddr
	nop
	nop
	nop
	halt
1:
#endif
	enable_int
	bic	AT, 3, AT
	jmp	zero, (AT), 0

/*------------------------------------------------------------------------------
** FUNCTION: pal_ret
**	Return from Kernel mode and continue at given point
**	Oposite part of kernel ()
**
** MODE: K
**
** INPUT PARAMETERS:  
**	AT  - register to continue in PALmode.
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	Enters PALcode.
*/
	CALL_PAL_PRIV (PAL_RETURN_ENTRY)
pal_retuser_entry:
	enable_int
	close_frame

	
/*------------------------------------------------------------------------------
** FUNCTION: pal_outstr_entry
**	prints a string to screen
**
** MODE: K
**
** INPUT PARAMETERS:
**	a0 :	offset of string
**	a1 :	screen base (in byte, 0, 1, 2, ...)
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	CALL_PAL_PRIV (KDBG_PRINT_HEX)
pal_outhex_entry:
	STALL
	STALL
	br	zero, pal_outhex_cont
	
	
	CALL_PAL_PRIV (KDBG_PRINT_STR)
pal_outstr_entry:
	STALL
	STALL
	br	zero, pal_outstr_cont


	CALL_PAL_PRIV (KDBG_PRINT_CHAR)
pal_outchar_entry:	
	STALL
	STALL
	br	zero, pal_outchar_cont

	CALL_PAL_PRIV (KDBG_GET_CHAR)
pal_getchar_entry:
	br	zero, pal_getchar_cont

	CALL_PAL_PRIV (KDBG_PRINT_DTB)
pal_print_dtb:
	br	zero, pal_print_dtb_cont	

	CALL_PAL_PRIV (KDBG_PRINT_ITB)
pal_print_itb:
	br	zero, pal_print_itb_cont	
	
/*==============================================================================
** UNPRIV PALCalls
**
** L4 System Service Procedures
**
**/
	.text	2
	. = 0x3000


	UNPRIV_UNDEF(0x80)
	UNPRIV_UNDEF(0x81)
	UNPRIV_UNDEF(0x82)
	UNPRIV_UNDEF(0x83)
	UNPRIV_UNDEF(0x84)
	UNPRIV_UNDEF(0x85)
	
/*------------------------------------------------------------------------------
** FUNCTION: l4_imb
**	synchronizes I-stream with cache
**	this function is an Alpha HW-required function
**	and some programs require it.
**
** MODE: U
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
**	cache misses will occur hereafter
*/

	CALL_PAL_UNPRIV (L4_IMB_ENTRY)
l4_imb:
	NOP
	br	zero, l4_imb_cont

	UNPRIV_UNDEF(0x87)
	UNPRIV_UNDEF(0x88)
	UNPRIV_UNDEF(0x89)
	UNPRIV_UNDEF(0x8a)
	UNPRIV_UNDEF(0x8b)
	UNPRIV_UNDEF(0x8c)
	UNPRIV_UNDEF(0x8d)
	UNPRIV_UNDEF(0x8e)
	UNPRIV_UNDEF(0x8f)

	UNPRIV_UNDEF(0x90)
	UNPRIV_UNDEF(0x91)
	UNPRIV_UNDEF(0x92)
	UNPRIV_UNDEF(0x93)
	UNPRIV_UNDEF(0x94)
	UNPRIV_UNDEF(0x95)
	UNPRIV_UNDEF(0x96)
	UNPRIV_UNDEF(0x97)
	UNPRIV_UNDEF(0x98)
	UNPRIV_UNDEF(0x99)
	UNPRIV_UNDEF(0x9a)
	UNPRIV_UNDEF(0x9b)
	UNPRIV_UNDEF(0x9c)
	UNPRIV_UNDEF(0x9d)
	UNPRIV_UNDEF(0x9e)
	UNPRIV_UNDEF(0x9f)

	UNPRIV_UNDEF(0xa0)
	UNPRIV_UNDEF(0xa1)
	UNPRIV_UNDEF(0xa2)
	UNPRIV_UNDEF(0xa3)
	UNPRIV_UNDEF(0xa4)
	UNPRIV_UNDEF(0xa5)
	UNPRIV_UNDEF(0xa6)
	UNPRIV_UNDEF(0xa7)
	UNPRIV_UNDEF(0xa8)
	UNPRIV_UNDEF(0xa9)
	UNPRIV_UNDEF(0xaa)
	UNPRIV_UNDEF(0xab)
	UNPRIV_UNDEF(0xac)
	UNPRIV_UNDEF(0xad)
	UNPRIV_UNDEF(0xae)
	UNPRIV_UNDEF(0xaf)
		
	
/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb0]: ipc
**
**	the sender always provides the data transfer, 
**	the receiver is passive, except irq message delivering
**
** MODE: U
**
** INPUT PARAMETERS:  
**
**      a0      - receiver id
**
**      a1      - snd_msg_vect
**
**      a2      - rec_msg_vect
**
**      a3      - msg_timeout = (receive = 63..32 | send = 31..00)
**
**	a4	- virtual sender id
**
**	a5	- original destination (for chief calls)
**
**      t0..t6, t8  - msg0 .. msg7
**
** OUTPUT PARAMETERS:  
**
**      v0 - message transfer result (full message dope)
**      pv - transferred dope
**
** SIDE EFFECTS:
**
**      scratch : t8..t12
**
*/ 
        
 	CALL_PAL_UNPRIV (L4_IPC_ENTRY)
l4_ipc:
#ifdef PERFORMANCE
	br	Xcnt_ipc
	nop
#endif	
XRcnt_ipc:
	open_frame					// Faults during touching TCBs
	ldiq	AT, TCB_MASK

	bic	sp, AT, p0				// better perf. for tcb
	bge	a1, ipc_has_send

	br	ipc_receive_only			// yes, jump to ipc_receive_only


	


/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb1]: fp_unmap
**
** MODE: U
**
** INPUT PARAMETERS:  
**	a0  - fpage
**	a1  - map mask
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	influences TLB's
**
*/
	CALL_PAL_UNPRIV (L4_FP_UNMAP_ENTRY)
l4_fp_unmap:
	STALL
	STALL
	br	mem_fp_unmap

Xcnt_ipc:
	ldiq	p1, CNT_BASE+CNT_IPC
	ldq_p	p0, 0(p1)
	addq	p0, 0x1, p0
	stq_p	p0, 0(p1)
	ldiq	p0, 0x00005410
	zap	p0, 0xf0, p0
	mtpr	p0, PmCtr
	br	XRcnt_ipc


/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb2]: id_myself
**
**	returns either thread's own ID or the next chief.
**	
** MODE: U
**
** INPUT PARAMETERS:  
**
**	a0 : NULL / task-id 
**
** OUTPUT PARAMTERES:
**
**	v0 : id
** 
** SIDE EFFECTS
**
*/
	CALL_PAL_UNPRIV (L4_ID_MYSELF_ENTRY)
l4_id_myself:
	STALL
	STALL
	blt	a0, id_myself_cont_ms
	br	zero, id_myself_cont_nc

/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb3]: switch_thread
**
**	switches to given thread or takes next from busy-queue
**
** MODE: U
**
** INPUT PARAMETERS:  
**
**	a0 : target thread
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
** 	Scratches ALL register, except SP
*/

	CALL_PAL_UNPRIV (L4_SWITCH_ENTRY)
l4_switch:  
	STALL
	STALL
	br	zero, l4_switch_cont



/*------------------------------------------------------------------------------
** FUNCTION: scheduler
**
** MODE: U
**
** INPUT PARAMETERS:  
**
**	a0 :	thread_id
**	a1 :	preempter
**	a2 :	parameter
**	
** OUTPUT PARAMTERES:
**	t0 :	partner_id
**	t1 :	old_preempter
**	t2 :	old_parameter
**	t3 :	time_slice
** 
** SIDE EFFECTS
**
*/

	CALL_PAL_UNPRIV (L4_SCHEDULE_ENTRY) 
l4_schedule:
	STALL
	STALL
	br	thread_schedule


/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb4]: lthread_ex_regs
**	Create a thread or modifies its data
**
** MODE: U
**
** INPUT PARAMETERS:  
**	a0	- lthread no
**	a1	- preempter
**	a2	- pager
**	a3	- pc
**	a4	- sp
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	CALL_PAL_UNPRIV (L4_LTHREAD_EX_REGS)
l4_lthread_ex_regs:
	STALL
	STALL
	br	lthread_ex_regs

/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb5]: l4_task_new
**	create and delete tasks
**
** MODE: U
**
** INPUT PARAMETERS:  
**
**	a0  - task id
**	a1  - new chief
**	a2  - pager id
**	a3  - intital IP
**
** OUTPUT PARAMTERES:
**
**	v0  - new task id
** 
** SIDE EFFECTS
**
*/
	CALL_PAL_UNPRIV (L4_TASK_NEW_ENTRY)
l4_task_new:
	STALL
	STALL
	br	task_new

/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb6]: l4_alpha_special
**	changes configuration of threads etc.
**
** MODE: U
**
** INPUT PARAMETERS:  
**	a0 - code
**	  0 - setidt
**		a1 - idt address
**	  1 - 
**
** OUTPUT PARAMTERES:
**
** SIDE EFFECTS
**
*/
	CALL_PAL_UNPRIV (L4_CONFIG_ENTRY)
l4_config_entry:
	STALL
	STALL
	br	zero, l4_config_cont


/*------------------------------------------------------------------------------
** FUNCTION [pal 0xb7]: l4_return
**	return to address ontop of stack
**
** MODE: U
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	all registers preserved !
**	user stack modified
*/
	CALL_PAL_UNPRIV (L4_RETURN_ENTRY)
l4_return:
	ldq	p1, 0(sp)
	addq	sp, 8, sp
	bic	p1, 3, p1
	mtpr	p1, excAddr
	STALL
	STALL
	hw_rei_stall	

		
/*******************************************************************************
** FUNCTION: user level debuging stuff ...
**
** MODE: U / K / P
**
** STATUS:
**	to be removed in final version
**
*/

	
/*------------------------------------------------------------------------------
** FUNCTION: pal_outstr_entry
**	prints a string/hex/char to screen
**
** MODE: U
**
** INPUT PARAMETERS:
**	a0 :	offset of string
**	a1 :	screen base (in byte, 0, 1, 2, ...)
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/

	CALL_PAL_UNPRIV (UDBG_PRINT_HEX)
udbg_outhex_entry:
	STALL
	STALL
	br	zero, pal_outhex_cont
	

	CALL_PAL_UNPRIV (UDBG_PRINT_STR)
udbg_outstr_entry:
	STALL
	STALL
	br	zero, pal_outstr_cont


	CALL_PAL_UNPRIV (UDBG_PRINT_CHAR)
udbg_outchar_entry:	
	STALL
	STALL
	br	zero, pal_outchar_cont

	CALL_PAL_UNPRIV (UDBG_GET_CHAR)
udbg_getchar_entry:
	STALL
	STALL
	br	zero, pal_getchar_cont
	

/*
**	Unprivileged halt procedure
*/
	CALL_PAL_UNPRIV (UDBG_SYSHALT)
udbg_halt:		
	STALL
	STALL
	br	pal_halt_cont


/*
** Unprivileged debug proc
*/
	CALL_PAL_UNPRIV (UDBG_DEBUG)
udbg_debug:	
	STALL
	STALL

	NewCom2SerialPutChar(0x64,p6,p7) //d
	
	open_frame
	disable_int
	
	mfpr	p1, excAddr
	STALL
	STALL
	mtpr	p1, tr0
		
	push	p_gp|p_AT
	kernel	debug_console


	
/*==============================================================================
** CONTINUATION Area
**
** All procedures with an fixed entry continues hereafter, if they need
** more space than available between the entry points
**
*/


/*------------------------------------------------------------------------------
** FUNCTION: pal_reset_cont
**	contiuation of reset code, initialze the chip
**
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS:	
**
*/
	
pal_reset_cont:
        extbl   a3, 3, a3                       // Get ........xx...... of Address

	IF	a3
		ldah    a0, (P_PALBASE >> 16)(zero)
	        subq    zero, 1, a3                     // Indicate, we are from boot
		call_pal 10                             // establish L4 PAL OSF_SWPPAL / ARC_SWPPAL

	XENDIF
	
	mtpr	zero, itbIa			// Flush the ITB.
	mtpr	zero, dtbIa			// Flush the DTB.

	
	ldiq	t0, P_PALBASE
	mfpr	t1, palBase

	mb
	mb
	NOP
	mtpr	t0, palBase
	mb
	mb
	mtpr	t1, ptPrevPal
	mtpr    zero, astrr			// Ignore AST interrupt requests.
        mtpr    zero, aster			// Disable AST interrupts.

        mtpr    zero, sirr			// Ignore software interrupt requests.
	ldah	t0, ICSR_K_INIT(zero)		// Get initial ICSR value.
#ifdef PERFORMANCE 
	lda	t0, 0x0300(t0)			// Enable Performance Counters
#endif
	
#if !defined(DISABLE_CRD)

	bis	zero, 1, v0			// Get a '1'
	sll	v0, ICSR_V_CRDE, v0		// Shift it into correct bit position
	or	v0, t0, t0			// Set the CRD enable bit

#endif /* DISABLE_CRD */
        mtpr    t0, icsr			// Initialize ICSR -- enable shadow!

        br      zero, set_icsr

        ALIGN_BLOCK
/*
** Since we modify SPE, we will follow directions and flush the icache.
*/
set_icsr:
        NOP
       	mtpr	zero, icFlush			// Flush the icache.
	NOP
	NOP

	NOP					// Required NOPs ... 1-10
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 11-20
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 21-30
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 31-40
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 41-44
	NOP
	NOP
	NOP

	lda	t0, (2<<MCSR_V_SP)(zero)	// Get a '10' (binary) in MCSR<SP>
	mtpr	t0, mcsr			// Set the super page mode enable bit
	lda	t0, DC_M_ENA(zero)		// Get a '1' in DC_MODE<ENA>
	mtpr	t0, dcMode			// Set the Dcache enable bit
	NOP

	STALL					// No Mbox instructions in 1,2,3,4
	STALL
	STALL
	STALL

	mtpr	zero, dcFlush			// Flush the Dcache

	disable_int t0				// Set internal IPL=1F

	mtpr	zero, ips			// Set Ibox mode to kernel	
	mtpr	zero, dtbCm			// Set Mbox mode to kernel

	mtpr	zero, excSum			// Clear any pending exceptions
	mfpr	zero, va			// Unlock the Mbox
 
	lda	t0, ICPERR_M_ALL(zero)
	mtpr	t0, icPerr			// Clear Icache parity errors & timeout
	lda	t0, DCPERR_M_ALL(zero)	
	mtpr	t0, dcPerr			// Clear Dcache parity errors

	mtpr	zero, dtbAsn
	mtpr	zero, itbAsn

	ldiq	t0, 0x67
	sll	t0, HWINT_V_PC0C, t0
	mtpr	t0, hwIntClr			// Clear any hardware int requests
	
	mtpr	zero, mafMode			// Clear miss address file mode

	ldiq	t0, P_PTBASE | 62 
	mtpr	t0, ptPageBase


	bsr	ra, sys_reset

	mtpr   zero, altMode			// set altMode to valid state

	ldiq	t1, 4096			// Cache prefetching for kernel !
	lda	t0, palcode			// increases performance 
	REPEAT
	  	ldq_p	t3, 0(t0)
		ldq_p	t4, 8(t0)
		addq	t0, 16, t0
		subq	t1, 1, t1
		CONT	t1
	ENDR

	stq_p	zero, PERF0(zero)
	subq	zero, 1, t0
	
	stq_p	t0, PERF1(zero)
	stq_p	t0, PERF2(zero)

	stq_p	zero, PERF3(zero)

	ldiq	p0, CNT_BASE
	ldiq	p1, 32

	REPEAT	
		stq_p	zero, 0(p0)
	
		subq	p1, 1, p1
		addq	p0, 8, p0
		CONT	p1
	ENDR

	pal_addr p1, pal_halt_handler		// Set emergency-entry if something fails
	bis	p1, 1, p1
	mtpr	p1, excAddr
	STALL
	STALL
		
	br	startup_dispatcher

	
/*------------------------------------------------------------------------------
** FUNCTION: startup-dispatcher
**	
** MODE: P(i)
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	creates stack for the dispatcher
**	jumps to kernel mode
*/

	.align	4
dispatcher_id:		
	.quad	0x302e3453504f5244
	
startup_dispatcher:
	tcb_dispatcher v0			// Set the kernel stack to the end of the first TCB
						// Which is the dispatcher

	lda	sp, TCB_STACKTOP(v0)		// size of TCB
	mtpr	sp, ptKSP			// Set KSP

	and	sp, sp, a0
	ldiq	p1, 0x7c
		bis	a0, a0, p3
		REPEAT
			subq	p1, 4, p1
			srl	p3, p1, a0
			and	a0, 0x0f, a0
			cmpule	a0, 9, p0
			cmovne	p0, 48, p0
			cmoveq	p0, 87, p0
			addq	a0, p0, a0
			NewCom2SerialPutCharReg(a0,p6,p7)

			CONT	p1
		ENDR
		bis	p3, p3, a0
	
	
//	clr	t0
//	push	p_t0
//	push	p_t0

	bis	zero, TCS_USED, t2
	stl	t2, TCB_COARSE_STATE (v0)	// Set Used bit in coarse state

	bis	zero, TFS_RUNNING, t2
	stq	t2, TCB_THREAD_STATE (v0)

	stq	v0, TCB_PRESENT_QUEUE (v0)	// present_queue_next -> myself
	stq	v0, (TCB_PRESENT_QUEUE+8) (v0)	// present_queue_prev -> myself

	stq	v0, TCB_BUSY_QUEUE (v0)		// Initialize busy queue
	stq	v0, (TCB_BUSY_QUEUE+8)(v0)

	krn_addr t1, system_cb			// system cb addresse
	
	stq	v0, SCB_BUSY_QUEUE (t1)		// root of busy queue points always to dispatcher, which
						// is this thread. VA -> DTB Miss -> Cache fill ! 

	lda	t1, TCB_SEND_ROOT (v0)
	stq	t1, (TCB_SEND_ROOT)(v0)
	stq	t1, (TCB_SEND_ROOT+8)(v0)
	
	pal_addr t2, dispatcher_id

	ldq_p	t2, 0(t2)
	stq	t2, TCB_MYSELF (v0)		// store myself id

	bis	zero, LLS_PRESENT_QUEUE | LLS_BUSY_QUEUE, t2
	stl	t2, TCB_LIST_STATE (v0)		// Set list state

	ldiq	t2, MCP_DISPATCHER		// Set dispatcher mcp
	stl	t2, TCB_SCHED_MCP (v0)
	
	ldiq	t2, SCP_DISPATCHER		// Set dispatcher scp
	stl	t2, TCB_SCHED_SCP (v0)

	pal_addr t1, thread_priorities		// ... and queue it in
	s4addq	t2, t1, t1
	stl_p	v0, 0(t1)

	subq	zero, 1, t2
	stq	t2, TCB_PAGER (v0)

	mtpr	zero, ptCurrentTime		// clear time register
	mtpr	zero, ptCurrentTicks		// clear time register	

	
	kernel	l4_kernel_init

/*------------------------------------------------------------------------------
** FUNCTION:
**	various interrupt handlers
**
*/

	ALIGN_BRANCH
pal_halt_handler:
	sys_halt 0xdead
	
	STALL
	STALL
	open_frame
	disable_int
	
	push	p_gp|p_AT
	kernel	debug_console
	
	ALIGN_BRANCH
pal_passive_release:	
	nop
	nop	
	hw_rei					// Back to user ...


pal_dfault_cont:
	open_frame

	push	p_r0|p_r1|p_r2
	
	ldq_p	p0, 0x60(zero)
	addq	p0, 1, p0
	stq_p	p0, 0x60(zero)
	tcb	p1
	stq_p	p1, 0x68(zero)

	nop
	nop

	mfpr	r1, va
	mfpr	r2, excAddr
	mfpr	r3, mmStat

	kmsg	"Invalid access 0x105"
	debug
	NOP
	NOP
	NOP
	pop	p_r0|p_r1|p_r2
	
	close_frame
		
	NOP
	NOP
	push	p_r1
	mfpr	r1, va
	mfpr	r2, excAddr
	mfpr	r3, mmStat
	debug
	
	sys_halt 0x6666	

	

/*------------------------------------------------------------------------------
** FUNCTION: itb_tlb_miss
**	PAL ITB Miss handler continuation area 
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	ALIGN_BRANCH
itb_protection_miss:

	ldiq	p6, CNT_BASE+CNT_ITLB_MISS
	ldq_p	p7, 0(p6)
	addq	p7, 1, p7
	stq_p	p7, 0(p6)

	// Depending on t1 bits error handling
	bic	p1, 0x40, p1			// PP_INTREE is always set
	bne	p1, itb_protection_fault
itb_translation_valid:

	bne	p4, itb_guard_fault

	bic	p0, 0xff, p0
	IFLT	p0
		or	p0, 0x10, p0		// Set ASM bit
	ENDIF
	mtpr	p0, itbPte			// Page Table Entry
	mtpr	p3, itbTag			// Virt. Address

	hw_rei_stall

	ALIGN_BRANCH
itb_protection_fault:

	mov	p2, p4
	open_frame
	push	p_a0|p_a1|p_a2|p_AT|p_gp

	bis	p3, p3, a0			// faulting address
	bis	p4, p4, a1			// last page table address
						// Since we're going straight to kernel, we don't need
						// to reload px registers
	pf_stack
	kernel	l4_itb_protection_fault


	ALIGN_BRANCH	
itb_guard_fault:

	ldiq	p6, CNT_BASE+CNT_ITLB_MISS
	ldq_p	p7, 0(p6)
	addq	p7, 1, p7
	stq_p	p7, 0(p6)

	mov	p2, p4
	open_frame
	push	p_a0|p_a1|p_a2|p_AT|p_gp

		
	bis	p3, p3, a0			// faulting address
	bis	p4, p4, a1			// last page table address
	
						// Since we're going straight to kernel, we don't need
						// to reppush px registers
	pf_stack
	kernel	l4_itb_guard_fault





/*------------------------------------------------------------------------------
** FUNCTION:
**	PAL DTB Miss handler continuation area 
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/

	ALIGN_BRANCH
dtb_protection_miss:
	// Depending on t1 bits error handling
	bic	p1, 0x40, p1			// PP_INTREE is always set
	bne	p1, dtb_protection_fault
	
dtb_translation_valid:
	bne	p4, dtb_guard_fault

	bic	p0, 0xff, p0
	IFLT	p0
		or	p0, 0x10, p0		// Set ASM Bit
	ENDIF
	
	mtpr	p0, dtbPte			// Page Table Entry
	mtpr	p3, dtbTag			// Virt. Address
	mfpr	p0, tb0				// Reload Scratch Register
	mfpr	p1, tb1
	mfpr	p2, tb2
	mfpr	p3, tb3
	mfpr	p4, tb4
	mfpr	p5, tb5
	mfpr	p6, tb6
	mfpr	p7, tb7
	STALL
	hw_rei_stall	


/*
** Before we go to kernel, we need to restore the p# regs, because they'll be
** scratched due to pal->kernel crossing
*/	
dtb_protection_fault:
	mov	p2, p4
	open_frame

	mfpr	p0, tb0				// Reload Scratch Register
	mfpr	p1, tb1
	push	p_p0|p_p1
	
	push	p_a0|p_a1|p_a2|p_AT|p_gp
		
	ldq_a	a1, 0x38(sp)

	mov	p3, a0				// faulting address
	
						// Since we're going straight to kernel, we don't need
						// to reload px registers
	mfpr	a2, mmStat

	pf_stack
	kernel	l4_dtb_protection_fault
	



/*------------------------------------------------------------------------------
** FUNCTION: dtb_guard_fault
**	jumped here after parsing the GPT tree
**	detects faulting address and checks for special kernel
**	region
**
** SIDE EFFECTS:	
**	continues in kernel mode
*/
	ALIGN_BRANCH
dtb_guard_fault:
	extll	p3, 4, p0			// extract the higher 32 bits
	addl	p0, 1, p0			// p0 is now 0, if p3 was 0xffffffffxxxxxxxx
	IF	p0
dtb_guard_fault_user:	
		mov	p2, p4
	
		open_frame
	
		mfpr	p0, tb0				// Reload Scratch Register
		mfpr	p1, tb1
		push	p_p0|p_p1
	
		push	p_a0|p_a1|p_a2|p_AT|p_gp

		mov	p3, a0		// faulting address
		ldq_a	a1, 0x38(sp)
		mfpr	a2, mmStat		// Since we're going straight to kernel, we don't need

						// to restore px registers
		pf_stack
		kernel	l4_dtb_guard_fault
	XENDIF
						// DTB Fault in kernel space
	extbl	p3, 3, p0			// Extract bits 24..31
	cmplt	p0, 0xe0, p1			//
	beq	p1, dtb_tcb_area		// >= 0xffff.ffff.e000.0000

	cmplt	p0, 0xd0, p1			// same functionalty !!!
	beq	p1, dtb_tcb_area		// >= 0xffff.ffff.d000.0000

	sys_halt	0xbbb
	cmplt	p0, 0x90, p1
	beq	p1, dtb_unknown			// >= 0xffff.ffff.9000.0000

	cmplt	p0, 0x10, p1
	beq	p1, dtb_ipc_area		// >= 0xffff.ffff.1000.0000

	debug
	br	zero, pal_halt_cont		// between 0xffff.ffff.0000.0000 and 0xffff.ffff.1000.0000
	
dtb_tcb_area:
	mov	p2, p4
	open_frame

	mfpr	p0, tb0				// Reload Scratch Register
	mfpr	p1, tb1
	push	p_p0|p_p1
	
	push	p_a0|p_a1|p_a2|p_AT|p_gp

	bis	p3, p3, a0			// faulting address
	bis	p4, p4, a1			// last page table address
	
						// Since we're going straight to kernel, we don't need
						// to reload px registers
	mfpr	a2, mmStat

	pf_stack
	kernel	l4_dtb_tcb_fault
	
	
dtb_unknown:
	ldiq	v0, 0x44444444
	br	zero, pal_halt_cont		// between 0xffff.ffff.0000.0000 and 0xffff.ffff.1000.0000	
	halt
	
dtb_ipc_area:
	open_frame

	ldiq	p1, 0x7fffff
	and	p3, p1, p2			// offset into region

	bis	p3, p3, p7

	tcb	p0

	ldq_a	p0, TCB_COMM_PARTNER (p0)			// Partner
	ldq_a	p3, TCB_COMM_ADDRESS (p0)			// Partner's base
	bic	p3, p1, p3
	addq	p3, p2, p3

	
	ldiq	p6, PP_WRITABLE | PP_INTREE			// PP_INTREE protection required

	ldiq	p4, 0x3ff 
	ldiq	p5, 0x1c0

	bis	p6, PP_INTREE, p6	
	
	task_nr	p0, p0						// t0 - task_nr
	pal_addr p1, pal_ptroots				// root pointer of ptabs
	s8addq	p0, p1, p0					// page base pointer to.

	ldl_p	p0, 0(p0)					// t0 = PageBase		
	bic	p3, p4, p3					// 0x3ff - clear protection area in address
	bis	p3, p5, p3					// XOR inverts only X, W, P bits 
	bic	p0, 0x3f, p2					// t2 =	old.frame start base
	srl	p3, p0, p1					// t1 = table_bits (a0)
	REPEAT	
		s4addq	p1, 0, p1				// t1 = t1 * 4	
		s4addq	p1, p2, p2				// t1 = t1 * 4 + t2 -- t1 * 16 + 4
		ldq_p	p1, 0(p2)				// load lower part of gpte
		ldq_p	p0, 8(p2)				// t0 = higher part of guard
		xor	p3, p1, p3				// guard bits and flip prot.
		bic	p0, 0x3f, p2				// t2 = old.frame start base	
		srl	p3, p1, p4				// shifted guard
		and	p3, p6, p1				// P - check
		bis	p3, p5, p3				// XOR inverts only X, W, P bits 
		bne	p1, dtb_ipc_protection_miss		// P - check
		srl	p3, p0, p1				// t1 = table_bits (a0)
		CONTZ	p4
	ENDR
	br	zero, dtb_ipc_guard_fault
 
	ALIGN_BRANCH
dtb_ipc_protection_miss:
        // Depending on t1 bits error handling
	bic	p1, 0x40, p1			// PP_INTREE is always set
	bne	p1, dtb_ipc_protection_fault

dtb_ipc_translation_valid:
	bne	p4, dtb_ipc_guard_fault

	bic	p0, 0xff, p0
	IFLT	p0
		or	p0, 0x10, p0		// Set ASM bit
	ENDIF

	
	mtpr	p0, dtbPte			// Page Table Entry
	mtpr	p7, dtbTag			// Virt. Address

	mfpr	p0, tb0				// Reppush Scratch Register
	mfpr	p1, tb1
	mfpr	p2, tb2
	mfpr	p3, tb3
	mfpr	p4, tb4
	mfpr	p5, tb5
	mfpr	p6, tb6
	mfpr	p7, tb7
	close_frame
	

dtb_ipc_protection_fault:
	NOP
	NOP
	debug
	br	pal_halt_cont
	
		
dtb_ipc_guard_fault:
	NOP
	NOP
	halt

	
/*------------------------------------------------------------------------------
** FUNCTION: dtb_double_guard_fault
**	jumped here after parsing the GPT tree
**	detects faulting address and checks for special kernel
**	region, but never map an address - simple map an zero page
**
** SIDE EFFECTS:	
**	return register MUST be r0 !!!!
*/
	ALIGN_BRANCH
dtb_double_guard_fault:
	extll	p3, 4, p0			// extract the higher 32 bits
	addl	p0, 1, p0			// p0 is now 0, if p3 was 0xffffffffxxxxxxxx
	bne	p0, dtb_guard_fault_user

#ifdef DEBUG
	mfpr	p0, mmStat			// Debug Check ! If ra != v0 then error
	srl	p0, MMSTAT_V_RA, p0
	and	p0, 0x1f, p0
	IF	p0
		sys_halt 0x4444
	ENDIF
#endif
	clr	v0				// clear v0
	
	mfpr	p0, tb0
	mfpr	p1, tb1
	mfpr	p2, tb2
	mfpr	p3, tb3
	mfpr	p4, tb4
	mfpr	p5, tb5
	mfpr	p6, tb6
	mfpr	p7, tb7
	STALL
	hw_rei_stall	
			

/*------------------------------------------------------------------------------
** FUNCTION: ret_(d/i)tb_*
**	various itb / dtb return handler.
**
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	ALIGN_BRANCH
ret_itb_fill:
	bic	a1, 0xff, a1
	IFLT	a1
		or	a1, 0x10, a1		// Set ASM bit
	ENDIF
	
	mtpr	a1, itbPte
	nop
	
	mtpr	a0, itbTag
	nop
	
	nop
	nop

	nop
	nop

	nop
	nop

	pf_restack

	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	close_frame

ret_itb_schedule:
	pf_restack
	
	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	close_frame

/*------------------------------------------------------------------------------
** FUNCTION: ret_(d/i)tb_*
**	various itb / dtb return handler.
**	ITB and DTB handler differs since DTB saves some Shadow Registers...
**		
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	
ret_itb_rerun:
	pf_restack
	
	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	close_frame
	
	
	ALIGN_BRANCH
ret_dtb_fill:
	bic	a1, 0xff, a1
	IFLT	a1
		or	a1, 0x10, a1		// Set ASM bit
	ENDIF
	
	mtpr	a1, dtbPte
	nop
	
	mtpr	a0, dtbTag
	nop


	nop
	nop

	nop
	nop

	nop
	nop	

	STALL

	pf_restack
	
	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	pop	p_p0|p_p1
	
	close_frame				// Scratches p0 and p1 !!

	ALIGN_BRANCH
ret_dtb_schedule:
	pf_restack
	
	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	pop	p_p0|p_p1
	
	close_frame

	ALIGN_BRANCH
ret_dtb_rerun:
	pf_restack
	
	pop	p_a0|p_a1|p_a2|p_AT|p_gp
	pop	p_p0|p_p1
	close_frame



	
/*------------------------------------------------------------------------------
** FUNCTION: pal_halt_cont
**	Halt continuation procedure, cont. at system specific halt
**
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	ALIGN_BRANCH
pal_halt_cont:
	br	zero, sys_halt



pal_opcdec_cont:
	STALL
	STALL
	open_frame
	disable_int
	
	mfpr	p1, excAddr
	ldiq	p0, CNT_BASE+CNT_EXCADDR
	STALL
	STALL
	stq_p	p1, 0(p0)
	
	push	p_gp|p_AT
	kernel	debug_console
	
	sys_halt	0x18

	

/*------------------------------------------------------------------------------
** FUNCTION: pal_outhex_cont
**	continues the output of a 64 bit hex value
**
*/

pal_outhex_cont:
	open_frame
	STALL
	STALL
	STALL
	bis	ra, ra, p4
	
	and	a1, 0x7c, p1
	IF	p1
		bis	a0, a0, p3
		REPEAT
			subq	p1, 4, p1
			srl	p3, p1, a0
			and	a0, 0x0f, a0
			cmpule	a0, 9, p0
			cmovne	p0, 48, p0
			cmoveq	p0, 87, p0
			addq	a0, p0, a0
	
			bsr	ra, sys_putchar
			CONT	p1
		ENDR
		bis	p3, p3, a0
	ENDIF			
	bis	p4, p4, ra
	close_frame
	
	
/*------------------------------------------------------------------------------
** FUNCTION: pal_outstr_cont
**	continues the output of a string given in a0 to screen addr a1
**
*/
	ALIGN_BRANCH
pal_outstr_cont:
	open_frame
	
	push	p_ra
	bsr	ra, sys_cwrite
	pop	p_ra
	close_frame

pal_outchar_cont:
	open_frame
	bis	ra, ra, p4
	bsr	ra, sys_putchar
	bis	p4, p4, ra
	close_frame

pal_getchar_cont:
	open_frame
	bis	ra, ra, p4
	bsr	ra, sys_getchar
	bis	p4, p4, ra
	close_frame
	
pal_print_dtb_cont:			
	open_frame
	bis	ra, ra, p4
	bsr	ra, debug_dtb
	bis	p4, p4, ra
	close_frame

pal_print_itb_cont:			
	open_frame
	bis	ra, ra, p4
	bsr	ra, debug_itb
	bis	p4, p4, ra
	close_frame

	
 

/*------------------------------------------------------------------------------
** FUNCTION: id_myself_cont_nc / id_myself_cont_ms
**	returns the thread id of a thread, or its chief
*/
	ALIGN_BRANCH
id_myself_cont_nc:
	open_frame				// really needed at this point, I don'y believe ?	
	nop
	mov	zero, a5			// WAS (danielp) nop
						// this indicates non IPC call (harsh)
	nop
	kernel	l4_id_nearest

	ALIGN_BRANCH
id_myself_cont_ms:
	open_frame				// really needed at this point, I don'y believe ?	
	tcb	v0
	ldq_a	v0, TCB_MYSELF (v0)		// load id
	close_frame				

/*------------------------------------------------------------------------------
** FUNCTION: l4_switch_cont
**
*/
	ALIGN_BRANCH
l4_switch_cont:
	open_frame
	nop
	disable_int
	nop
	kernel	k_simple_switch



/*------------------------------------------------------------------------------
** FUNCTION: l4_imb_cont
**	Instruction Memory Barrier - Continuation
**
** MODE:
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**
*/
	ALIGN_BRANCH
l4_imb_cont:
        mb
       	mtpr	zero, icFlush			// Flush the icache.
	NOP
	NOP

	NOP					// Required NOPs ... 1-10
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 11-20
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 21-30
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 31-40
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP
	NOP

	NOP					// Required NOPs ... 41-44
	NOP
	NOP
	NOP
	mb
	hw_rei	


/*------------------------------------------------------------------------------
** Function L4_CONFIG_CONT
*/
	ALIGN_BRANCH
	
l4_config_cont:
	open_frame

	cmpeq	a0, CNF_SETIDT, p0
	IF	p0
		tcb	p0
		stq_a	a1, TCB_IDT (p0)
	ENDIF
	close_frame	

	
	
/*------------------------------------------------------------------------------
** Interrupt Table Handler
**
**	p0 :	 Interrupt number
*/
	ALIGN_BRANCH
l4_do_exception:
	mfpr	p2, excAddr
	mfpr	p1, ips				/* Kernel / User Mode */
	nop
	nop
	nop
	IFLBC   p2
		IF	p1				/* p1 != 0, if from user mode */
			mfpr	p1, ptKSP
			nop
			ldiq	p3, TCB_MASK
			bic	p1, p3, p1
			ldq_a	p3, TCB_IDT(p1)
			IF	p3
				subq	sp, 8, sp
				stq	p2, 0(sp)	/* Push user stackpointer		*/
				s8addq	p0, p3, p3	/* p3 = p3 + 8 * p0 => idt entry	*/
				ldq	p3, 0 (p3)	/* Load interrupt entry			*/
				bic	p3, 3, p3	/* Clear lower bits			*/
				mtpr	p3, excAddr
				STALL
				STALL
				hw_rei
			ENDIF
		ENDIF
	ENDIF
//	br	pal_halt_cont
	open_frame
	tcb	t0
	ldq_a	t1, TCB_IDT(t0)
	khex	t1, 64
	debug
	br	pal_halt_cont				/* Shit happens, error in PAL/Krn mode occured */
	close_frame
	
#if 0
// seedy debugging hack not needed now.
debug_special:	
	mfpr	t10, va
	mfpr	t9, excAddr
	nop
	nop
	debug
	halt	
#endif	
		
/*==============================================================================================
** PAL + Kernel data area
*/

	ALIGN_BLOCK
system_cb:
	.fill	SIZE_SCB, 1, 0				// size of SCB, filled with 0

	ALIGN_BLOCK
interrupt_queue:
	.fill	MAXINT*2, 8, 0				// 32 IRQ, a double linked list

	ALIGN_BLOCK
interrupt_owner:					// 32 IRQ ownerships
	.fill	MAXINT, 8, 0

	ALIGN_BLOCK
thread_priorities:	
	.fill	MAX_PRIOS, 4, 0
	
	
/*==============================================================================================
** Kernel page table
** used to get access to the PALcode via virtual addresses.
** the last page at 0x21e000 is used as first tcb page, and TCB (0,0) as stack for
** the kernel startup...
** Entries for kernel area are always shared. We don't need unshared, or task specific entries
*/
#define GPT(h,l,fr)	\
	.long	l;	\
	.long	h;	\
	.quad	fr

#define GPT0		\
	.quad	0,0

#define ASM	(1<<63)	

	.text	4
	. = O_PTBASE

pal_ptbase:	
	//---------------------------------------------------------------
	GPT0
	GPT0	// never used on 43 bit Alpha
	GPT0	// never used on 43 bit Alpha
	GPT (0xffffffff, 0x00000000 | (PP_KERNEL | PP_EXECUTABLE | PP_WRITABLE | PP_INTREE | (64-32)), 0x00208100 | (64 - 36))

	//---------------------------------------------------------------
	. = 0x8100
	GPT (0x00000000, 0x00200000 | (PP_EXECUTABLE | PP_WRITABLE | PP_INTREE | (64-47)), 0x00208200 | (64 - 51))		// Kenel
	GPT0	// ... Start of Communication Area
	GPT0
	GPT0

	GPT0
	GPT0
	GPT0
	GPT0

	GPT0
	GPT0	// ... End of CA
	GPT0
	GPT0

	GPT0
	GPT0

	GPT (0x00000000, 0xE0000000 | (PP_WRITABLE | PP_INTREE | (64-36)), 0x0020A000 | (64 - 44))		// TCB Space
	GPT (0x00000000, 0xF0000000 | (PP_WRITABLE | PP_INTREE | (64-36)), 0x0020C000 | (64 - 44))		// TCB Space

	//--- Kernel -----------------------------------------------------------
	GPT (0x00000000, 0x00000000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00200000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00002000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00202000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00004000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00204000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00006000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00206000) >> 13) << 32))	// Phys Addr	

	GPT (0x00000000, 0x00008000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00208000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0000A000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0020A000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0000C000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0020C000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0000E000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0020E000) >> 13) << 32))	// Phys Addr	

	GPT (0x00000000, 0x00010000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00210000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00012000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00212000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00014000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00214000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x00016000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00216000) >> 13) << 32))	// Phys Addr	

	GPT (0x00000000, 0x00018000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x00218000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0001A000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0021A000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0001C000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0021C000) >> 13) << 32))	// Phys Addr	
	GPT (0x00000000, 0x0001E000 | PP_EXECUTABLE | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0021E000) >> 13) << 32))	// Phys Addr	


	GPT0
	GPT0
	GPT0

	GPT0
	GPT0
	GPT0
	GPT0

	GPT0
	GPT0
	GPT0
	GPT0

	GPT0
	GPT0
	GPT0
	GPT0

	//--- TCB --------------------------------------------------------------
	.= 0x9000
	GPT (0x00000000, 0x00000000 | PP_WRITABLE | 13, ASM | 0x00001100 | (((0x0021e000) >> 13) << 32))	// Phys Addr	21
	
	.= 0x9400
	.= 0x9800
	.= 0x9C00

	.= 0xA000
	GPT (0x00000000, 0x00000000 | (PP_WRITABLE | PP_INTREE | (64-45)), 0x00209000 | (64 - 51))	
	
		
/*==============================================================================
** Page Roots...
** If we need a root page table entry, we look at this table
** the entry is one long for the table, one long for the map tree. 0 in the map-tree
** means no tree !
*/

	. = 0xc000
pal_ptmaps:	
	.fill	(MAX_TASKS), 8, 0	
	
	.text	0x0e
	. = O_PTROOTS
pal_ptroots:
	.quad	(P_PTBASE | (64-2))
	.quad	0x00000000e0000000
	.fill	(MAX_TASKS-2), 8, 0x00000000e0080000


/*=============================================================================
** End OF PAL code at offset 0x10000 !
*/


	
.END

