/*
**	Copyright (c) 1995, Sebastian Schoenberg 
**	E-mail : ss10@irz.inf.tu-dresden.de
**
**	University of Technology Dresden, Department of Computer Science
**	Inistute for Operating Systems, Databases and Computer Networks
**
**	$Id: memory.S,v 5.2 1999/11/09 00:42:15 danielp Exp $
**
*/

#if !defined(lint)
	.data
	.asciz	"$Id: memory.S,v 5.2 1999/11/09 00:42:15 danielp Exp $"
#endif

//#define BIGMEM256	

#include <pal/predef.h>
	
#include <pal/dc21164.h>
#include <pal/regdef.h>
#include <pal/macros.h>

#include <pal/l4pal.h>

#include <pal/gasp.h>
	
	.text

	.globl	memory_init
	
	.globl	mem_translate_address
	
	.globl	mem_map_page
	.globl	mem_grant_page
	.globl	mem_flush_page
	.globl	mem_flush_tlbpte
	
	.globl	mem_grab_frame
	.globl	mem_free_frame
	.globl	mem_init_pool
	
	.globl	mem_insert_pte
	.globl	mem_fp_unmap	
	.globl	mem_delete_as
	
	
/*------------------------------------------------------------------------------
** FUNCTION: mem_translate_address
**	translate a virtual address into a physical 
**	address, based on the pageroot given indirectly by t0
**	an address can ever be translated, if t6 is only PP_INTREE 
**
** MODE: K
**	
** INPUT PARAMETERS:  
**
**	t3 -	virtual address
**	t0 -	tcb of task
**	t6 -	Page Protection
**	
** OUTPUT PARAMTERES:
**
**	a1 - result, 0 if fail
**	t7 - last page table ptr (or zero if in sigma 0)
** 
** SIDE EFFECTS
**	t0, t1, t2, t3, t4, t5, t6  scratched
*/
	ALIGN_BRANCH

mem_translate_address:
	push	p_t3
	
	ldiq	t4, 0x3ff 
	ldiq	t5, 0x1c0

	bis	t6, PP_INTREE, t6	
	
	task_nr	t0, t0				// t0 - task_nr
	push	p_t0
	pal_addr t1, pal_ptroots		// root pointer of ptabs
	s8addq	t0, t1, t0			// page base pointer to.

	ldl_p	t0, 0(t0)			// t0 = PageBase	
	bic	t3, t4, t3			// 0x3c0 - clear protection area in address
	bis	t3, t5, t3			// XOR inverts only X, W, P bits
	bic	t0, 0x3f, t2			// t2 = old.frame start base
	srl	t3, t0, t1			// t1 = table_bits (a0)
	REPEAT	
		s4addq	t1, 0, t1		// t1 = t1 * 4	
		s4addq	t1, t2, t2		// t1 = t1 * 4 + t2 -- t1 * 16 + 4
		ldq_p	t1, 0(t2)		// load lower part of gpte
		ldq_p	a1, 8(t2)		// t0 = higher part of guard
		mov	t2, t7			// preserve last page table entry ptr 
		xor	t3, t1, t3		// guard bits and flip prot
		bic	a1, 0x3f, t2		// t2 = old.frame start base	
		srl	t3, t1, t4		// shifted guard
		and	t3, t6, t1		// P - check
		bis	t3, t5, t3		// XOR inverts only X, W, P bits 
		bne	t1, mta_protection_miss	// P - check
		srl	t3, a1, t1		// t1 = table_bits (a0)
		CONTZ	t4
	ENDR
	br	mta_guard_fault
 
	ALIGN_BRANCH
mta_protection_miss:
        // Depending on t1 bits error handling
	bic	t1, 0x40, t1			# PP_INTREE is always set
	IFZ	t1
		bne	t4, mta_guard_fault
		pop	p_t0
		pop	p_t3
		ret	zero, (ra)
	XENDIF
	pop	p_t0
	pop	p_t3
	clr	a1
	ret	zero, (ra)

	ALIGN_BRANCH
mta_guard_fault:
	pop	p_t0					
	pop	p_t3
	cmpeq	t0, TASK_SIGMA0, t0			// Is this a sigma 0 address ?
	IF	t0
		extll	t3, 4, a1
		addl	a1, 1, a1
		IF	a1				// ffffffff.xxxxxxxx addresses don't belong to sigma 0
							// Ok, sigma 0 prot fault.
			ldiq	a1, P_SIGMA0BASE
			cmplt	t3, a1, a1

			clr	a1

			IFZ	a1  
				srl	t3, 13, a1		// PFN
				sll	a1, 32, a1		// insert PFN into a1
				ldiq	t0, 0x0ff00		// attributes
				or	t0, a1, a1
				clr	t7
				ret	zero, (ra)
			XENDIF
		ENDIF
	ENDIF
	clr	a1	
	ret	(ra)
		


/*------------------------------------------------------------------------------
** FUNCTION: mem_grab_frame
**	Grab a free frame and return its PHYSICAL address
**
** MODE: P
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
**	v0  - Physical Frame Address
**
** SIDE EFFECTS
**	t0 offset of impure area
**	scratches:	t0, AT, v0 
**	
*/
	ALIGN_BRANCH
mem_grab_frame:
	disable_int

	pal_addr t0, system_cb
	ldq_p	v0, SCB_FREE_FRAME (t0)
	IF	v0				// internal management error, no free memory
		ldq_p	AT, 0(v0)		// Load pointer to next frame
		stq_p	AT, SCB_FREE_FRAME(t0)
		ldiq	t0, 0x1fff
		and	AT, t0, AT
		IF	AT
			mov	AT, t0
			debug
		ENDIF
		stq_p	zero, 0(v0)
		enable_int
		ret	(ra)
	XENDIF
	enable_int
	kmsg	"Out of memory pool\n\r"
	debug
	br	mem_grab_frame

/*------------------------------------------------------------------------------
** FUNCTION: mem_free_frame
**	Frees a frame and return it into the free pool
**
** MODE: K
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
**	v0  - Physical Frame Address
**
** SIDE EFFECTS
**	clears the page
**	scratches :	 t0, AT
*/
	ALIGN_BRANCH
mem_free_frame:
	/* Clear the entire page */
	ldiq	t0, PAGESIZE
	mov	v0, AT
	REPEAT				
		stq_p	zero, 0x00(AT)
		stq_p	zero, 0x08(AT)
		stq_p	zero, 0x10(AT)
		stq_p	zero, 0x18(AT)
		subq	t0, 0x20, t0
		addq	AT, 0x20, AT
		CONT	t0
	ENDR
	
	pal_addr t0, system_cb
	disable_int
	ldq_p	AT, SCB_FREE_FRAME(t0)
	stq_p	AT, 0(v0)			// Store the old free address into the page
	stq_p	v0, SCB_FREE_FRAME(t0)		// Now insert into pool
	enable_int
	ret	(ra)
		


	
/*------------------------------------------------------------------------------
** FUNCTION: memory_init
**	Initialize Memory  --
**	check memory Size
**
** MODE: K
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
**	a0  - Top address of free pages
**	a1  - number of free pages
**
** SIDE EFFECTS
**	
*/

memory_init:	
	push	p_ra
	kmsg	"Testing main memory size "
	ldiq	a0, P_SIGMA0BASE
//	disable_int
	rpcc	t3

	mb
	mb

//	ldiq	a0, 0x10000000	// 256Mb (bug) - set max 128Mb

//	ldiq	a0, 0xfff0000  // 256Mb - a bit.

	bsr	ra, sys_whole_mem_size
	
//	ldiq	a0, 0x8000000  // 128Mb

	mb			// debug information..
	mb
	khex	a0, 64		// ..debug information
	
	enable_int

	lda	t0, system_cb
	stq	a0, SCB_SYSTEM_MEMORY (t0)
	srl	a0, 15, a1

	bsr	ra, mem_init_pool

	pop	p_ra
	ret	(ra)
	
/*------------------------------------------------------------------------------
** FUNCTION: mem_init_pool
**	Initialize Memory pool --
**	all pages in the pool are zeroed
**
** MODE: K
**
** INPUT PARAMETERS:  
**
** OUTPUT PARAMTERES:
**	a0  - Top address of free pages
**	a1  - number of free pages
**
** SIDE EFFECTS
**	
*/
	ALIGN_BRANCH
mem_init_pool:
	push	p_a0|p_a1
	push	p_a1
	push	p_a0
	kmsg	"Initializing Memory Pool ("
	pop	p_a0
	khex	a0, 32
	kmsg	","
	pop	p_a0
	khex	a0,32
	kmsg	")\n\r"
	pop	p_a0|p_a1

	mov	a1, t0
	mov	a0, t1
	clr	t2
	ldiq	t3, PAGESIZE
	REPEAT
		stq_p	t2, 0(t1)
		ldiq	t4, 1023
		addq	t1, 8, t5
		REPEAT
			stq_p	zero, 0(t5)
			addq	t5, 8, t5
			subq	t4, 1, t4
			CONT	t4
		ENDR
		mov	t1, t2
		subq	t1, t3, t1		// Go to previous page
		subq	t0, 1, t0
		CONT	t0
	ENDR
	
	pal_addr t1, system_cb
	stq_p	t2, SCB_FREE_FRAME (t1)
	ret	(ra)

	
			
/*------------------------------------------------------------------------------
** FUNCTION: mem_map_page
**	map a page and enter data into the maptree structure
**
** MODE: K
**
** INPUT PARAMETERS:
**	a0  - Virtual Address of Receiver
**	a1  - TLB frame data
**	t1  - Receiver TCB
**	t2  - Source pte *
**	t3  - virtual address of sender
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	scratches:	t0 .. t10
*/
	ALIGN_BRANCH
mem_map_page:
	push	p_ra
	push	p_t2					// Pointer to Source PTE
	push	p_t1|p_a0|p_a1
	mov	t1, t0
	clr	t6
	ldiq	a2, 13					// Page size
	ldiq	a4, PP_WRITABLE | PP_EXECUTABLE		// Page Attributes CHANGE IT
	disable_int
	bsr	ra, mem_insert_pte
	pop	p_t1|p_a0|p_a1

	pop	p_a2					// a2 Source PTE *

	
#if 1  // WUESTER HACK FUER VOLKMAR
	// if mapping is done in the same address space
	tcb	t0
	xor	t0, t1, t0
	srl	t0, TID_S_TASK, t0
	ldiq	a3, TID_M_TASK
	and	t0, a3, t0
	
	IFZ	t0					// Same task, refetch source PTE *
		push	p_t0|p_t1|p_t3|p_t5|p_t6|p_t7|p_t8
		mov	a0, t3				// virtual address
		tcb	t0
		ldiq	t6, PP_INTREE			// t6 required attribute
		bsr	ra, mem_translate_address
		mov	t7, a2				// Now the changed source PTE *
		pop	p_t0|p_t1|p_t3|p_t5|p_t6|p_t7|p_t8
	ENDIF
#endif	
	
	//						// a0 virtual address		
	mov	t9, a3					// a3 Target Page Table Ptr
	mov	t1, a4					// a4 Target TCB Address

	ldq_p	t1, 8(a3)
	IFZ	t1
		push	p_a0|p_a1
		kmsg	"Strange table ? Na und\n\r"
		khex	a0
		kmsg	" - "
		khex	a3
		kmsg	"\n\r"
		debug
		pop	p_a0|p_a1
		stq_p	zero, 0(a3)
		stq_p	zero, 8(a3)
		pop	p_ra
		ret	(ra)
	XENDIF

	// interrupts to be enabled when ready atomic changes.
	disable_int
	bsr	ra, mdb_add_mapping
	IFZ	v0
		pop	p_ra
		ret	(ra)	
	XENDIF
	stq_p	zero, 0(a3)
	stq_p	zero, 8(a3)
	pop	p_ra
	ret	(ra)
		
	
/*------------------------------------------------------------------------------
** FUNCTION: mem_grant_page
**	grant a page and enter data into the maptree structure
**
** MODE: K
**
** INPUT PARAMETERS:
**	a0  - Virtual Address of Receiver
**	a1  - TLB frame data
**	t1  - Receiver TCB
**	t2  - Source pte *
**	t3  - virtual address of sender
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	scratches:	t0 .. t10
*/
	ALIGN_BRANCH
mem_grant_page:	
	push	p_ra
	push	p_t2
	push	p_t1|p_t3|p_a0|p_a1
	mov	t1, t0
	clr	t6
	ldiq	a2, 13					// Page size
	ldiq	a4, PP_WRITABLE | PP_EXECUTABLE		// Page Attributes
	bsr	ra, mem_insert_pte
	pop	p_t1|p_t3|p_a0|p_a1

	pop	p_a2

	push	p_all
	//						// a0 virtual address		
	mov	t9, a3					// a3 Target Page Table Ptr
	mov	t1, a4					// a4 Target TCB Address
	disable_int
	bsr	ra, mdb_change_mapping
	pop	p_all
	

	stq_p	zero, 0(a2)
	stq_p	zero, 8(a2)
	//flush tlb in current addr space for ITB & DTB
	mtpr	t3, dtbIs
	mtpr	t3, itbIs

	pop	p_ra
	ret	(ra)	

/*------------------------------------------------------------------------------
** FUNCTION: mem_flush_tlbpte
**
**	flush an entry in TLB and Pptab
**
** MODE: K
**
** INPUT PARAMETERS:
**	a0  - virtual address
**	a1  - Task Number
**	a2  - Page Table Entry
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
*/
mem_flush_tlbpte:	
	stq_p	zero, 0(a2)
	stq_p	zero, 8(a2)
	//flush tlb in current addr space for ITB & DTB
	// set the ASN respectively
	
	//mtpr	a0, dtbIs
	//mtpr	a0, itbIs

	mtpr	zero, dtbIa		// brute force ...
	mtpr	zero, itbIa
	ret	(ra)

	

/*------------------------------------------------------------------------------
** FUNCTION: mem_flush_page
**	flush a page in tasks address space
**	is called from mdb function
**
** MODE: K
**
** INPUT PARAMETERS:
**	a0  - Virtual Address 
**	a1  - Task Number
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
*/
	ALIGN_BRANCH
mem_flush_page:
	push	p_ra
	
	push	p_a0|p_a1|p_t0|p_t1
	mov	t0, a0
	mov	t1, a1
	kmsg	"mem_flush_page is flushing from task:"
	khex	t1
	kmsg	" virtual addr:"
	khex	t0
	kmsg	"\n\r"
	pop	p_a0|p_a1|p_t0|p_t1
	
	pop	p_ra
	ret	(ra)




	
/*------------------------------------------------------------------------------
** FUNCTION: mem_insert_pte
**	insert a frame into the pagetable tree of the given addres
**	space. This may lead to splitting of PTE entries allocating
**	new frames and so on. So this is ABSOLUTELY not TIME 
**	predictable
**
** MODE: K
**
** INPUT PARAMETERS:  
**
**	t0 -	tcb of task
**	t6 -	Page Protection
**	
**	a0 -    virtual address	
**	a1 -	new TLB entry
**	a2 -	table size
**	a4 -	new protection
**	
** OUTPUT PARAMTERES:
** 
**	t9 - target page table entry ptr
**
** SIDE EFFECTS
**	scratches	t0, t1, t2, t3, t4, t5, t6, t7, t8, t9, t10, t11, AT, a1, a2
*/
	ALIGN_BRANCH
mem_insert_pte:
	ldiq	t4, 0x3c0 
	ldiq	t5, 0x1c0

	task_nr	t0, t0				// t0 - task_nr
	pal_addr t1, pal_ptroots		// root pointer of ptabs
	s8addq	t0, t1, t0			// page base pointer to.

	ldl_p	t0, 0(t0)			// t0 = PageBase		
	bic	a0, t4, t3			// 0x3c0 - clear protection area in address
	bis	t3, t5, t3			// XOR inverts only X, W, P bits 
	bic	t0, 0x3f, t2			// t2 = old.frame start base
	srl	t3, t0, t1			// t1 = table_bits (a0)
1:	
	s4addq	t1, 0, t1			// t1 = t1 * 4	
	s4addq	t1, t2, t2			// t1 = t1 * 4 + t2 -- t1 * 16 + 4
	mov	t2, t8				// keep this entry in t8
	ldq_p	t1, 0(t2)			// kpop lower part of gpte
	ldq_p	t7, 8(t2)			// t0 = higher part of guard


	xor	t3, t1, t3			// guard bits and flip prot.
	bic	t7, 0x3f, t2			// t2 = old.frame start base	
	srl	t3, t1, t4			// shifted guard
	and	t3, t6, t1			// P - check
	bis	t3, t5, t3			// XOR inverts only X, W, P bits 
	bne	t1, ipte_protection_miss	// P - check
	srl	t3, t7, t1			// t1 = table_bits (a0)
	beq	t4, 1b
	br	zero, ipte_guard_fault

	ALIGN_BRANCH
ipte_protection_miss:
        // Depending on t1 bits error handling
	bic	t1, 0x40, t1			# PP_INTREE is always set
	bne	t1, ipte_guard_fault

ipte_translation_valid:
	bne	t4, ipte_guard_fault
	ret	zero, (ra)			// Entry is always in tabls
	
	/*
	** Now we can think about inserting the Entry
	**
	**	t0 -  low part
	**	t1 -  high part
	**	t2 -  va_guard
	**	t3 -  va
	**	t4 -  va_remain
	**  	  t5 -  l1_new
	**	  t6 -  new_guard
	**	t8  - PTE of last parsing
	**
	**	t9 .. scratch
	**	a0 - virt address
	**	a1 - page frame
	**	a2 - table size
	**
	** Output 
	**	t9 - target page table entry ptr
	**
	*/
ipte_guard_fault:
	subq	a2, 4, a2

	ldq_p	t0, 0(t8)			// low part of this pte
	ldq_p	t1, 8(t8)			// high part

	ldiq	AT, 0x3ff
	bic	t3, AT, t2			// t2 = va_guard
	xor	t3, t0, t4			// t4 = va_remain
	// ** regs set **/
		
	or	t0, t1, AT			// AT = 0, if t0 == 0 and t1 == 0
	beq	AT, ipte_insert_empty
ipte_insert_split:
	push	p_ra

// new start
	ldiq	t9, 64
	mov	t3, t10
	
2:	blt	t10, 1f	
		subq	t9, 1, t9
		sll	t10, 1, t10
	br	2b	
1:	
	
	subq	t9, a2, AT
	and	t0, 0x3f, t10
	cmpult	t10, AT, AT			// AT = (l1 < dif_bit - table_size)
	IFZ	AT
		addq	t10, a2, t5		// if !AT:	l1_new = l1 + table_size
	ENDIF			
	cmovne	AT, t9, t5			// if AT:	l1_new = dif_bit
	srl	t0, t5, t9			
	sll	t9, t5, t6			// t6 = (old_guard >> l1_new) << l1_new = new_guard
	
	push	p_t0
	bsr	ra, mem_grab_frame		// grab an empty frame ZEROED
	pop	p_t0	
	beq	v0, ipte_nomem

	subq	t5, a2, t10			// t10 = l0_new = l1_new - table_size
	or	v0, t10, t10			// t10 = l0_new | new_tab
 	stq_p	t10, 8(t8)
	or	t6, t5, t10			// t10 = new_guard | l1_new
	or	t10, a4, t10			// t10 = t10 | protection	
	or	t10, PP_INTREE, t10
	stq_p	t10, 0(t8)			// store in low, so link the old entry

	// Calculate the entry data for the new frame
	
	xor	t4, t6, t10			// t10 = va_remain ^ new_guard
	subq	t5, a2, t11			// t11 = l0_new
	srl	t10, t11, t10			// t10 = (va_remain ^ new_guard) >> l0_new = new_index

	s4addq	t10, 0, t10
	s4addq	t10, v0, t10			// t10 = offset into table

	ldiq	t11, 64
	subq	t11, t5, t11			// t11 = 64 - l1_new

	sll	t4, t11, AT
	srl	AT, t11, t11			//   

	ldiq	AT, 0x3ff			
	bic	t11, AT, t11			// t11 = ((va_remain << (64 - l1_new)) >> (64 - l1_new)) & ~0x3ff;
	bis	t11, 13, t11			// t11 = t11 | PAGE size
	bis	t11, a4, t11			//                | protection

	stq_p	t11, 0(t10)			// store in low
	stq_p	a1, 8(t10)			// store new frame in high

	push	p_t10
	// Calculate the data for the old frame
	
	bic	t0, AT, t10			// t10 = old_guard

	xor	t10, t6, t10			// t10 = old_guard ^ new_guard
	subq	t5, a2, t11			// t11 = l0_new = (l1_new - table_size)
	srl	t10, t11, t9			// t9 = old_index =  (old_guard ^ new_guard) >> l0_new;

	bic	t10, AT, t10			// t10 = (old_guard ^ new_guard) & ~0x3ff = old_xguard;

	s4addq	t9, 0, t9
	s4addq	t9, v0, t9			// t9 = offset old into table

	
	
	ldiq	AT, 0x3c0
	and	t0, AT, AT			// AT = old_prot
	or	t10, AT, t10			// t10 = l1 | old_prot
	and	t0, 0x3f, AT			// AT = l1
	or	t10, AT, t10			// t10 = old_xguard | l1 | old_prot 
	
	stq_p	t10, 0(t9)			// store in low part
	stq_p	t1, 8(t9)			// store the old link.

	pop	p_t9
	
	pop	p_ra
	ret	zero, (ra)
	
ipte_insert_empty:
	mov	t8, t9
	bis	t2, 13, t2			// set guard` length
	or	t2, a4, t2
	stq_p	t2, 0(t9)			// kpush new low part
	stq_p	a1, 8(t9)			// Store new high part	
 	clr	a1
	ret	(ra)
		
ipte_nomem:
	kmsg	"Out of memory in PTSPLIT...\n\r"
	halt



/*------------------------------------------------------------------------------
** FUNCTION: mem_fp_unmap
**	unmaps a mapped flexpage
**	since no "explicit" mappings exist for Sigma 0,
**	nobody can unmap a page in Sigma 0, so it doesn't need to be
**	checked explicitely.
**
** MODE: K
**
** INPUT PARAMETERS:  
**	a0  - fpage
**
** OUTPUT PARAMTERES:
** 
** SIDE EFFECTS
**	t3 - running virtual address
**
*/
/*---------------------------------------------------------------------------------
** Unmap mapped or granted pages
*/
	ALIGN_BRANCH
mem_fp_unmap:	
        open_frame
        kernel  mem_fp_unmap_kern
	
	ALIGN_BRANCH
mem_fp_unmap_kern:
	push	p_ra

	srl	a0, 2, t1			// Extract number of pages
//	and	t1, 0x3f, t1			// 64-bit v-addr. (not yet)
	and	t1, 0x2b, t1			// 43-bit v-addr.
	subq	t1, PAGEBITS, t1

	IFGE	t1				// Size is at least 13, or greater
		ldiq	t3, 1
		sll	t3, t1, t1		// Number of Pages is 1 << (size - 13)
		ldiq	t3, PAGEMASK
		bic	a0, t3, t3		// Get Base Address
		REPEAT
			push	p_t1|p_t3
			tcb	t0
			clr	t6
			push	p_t3
			disable_int	// must be atomic
			bsr	ra, mem_translate_address
			pop	p_t3
			IF	a1
				bsr	ra, mdb_flush_mapping
			ENDIF
			pop	p_t1|p_t3
				
			ldiq	t2, PAGESIZE
			addq	t3, t2, t3	// Next Page

			subq	t1, 1, t1
			CONT	t1
		ENDR
	ENDIF
	pop	p_ra

	return_frame

/*------------------------------------------------------------------------------
** FUNCTION: mem_delete_tree (danielp)
**	Free up associated page table memory from the specified entry point.
**	This function is partically recursive.
**	
** MODE: K
**
** INPUT PARAMETERS:  
**	v0 - table pointer (frame pointer) to gpt table to traverse and free
**
** OUTPUT PARAMETERS:
**	nothing
** SIDE EFFECTS
**	scratches v0, t2, t3, t4, calls mem_free_frame
** C-CODE:
** mem_delete_tree(table_ptr) {	
**	for(i=0; i<PAGESIZE/PTE_ENTRYSIZE; i++) {
**		if(table_ptr[i].type != LEAF) {
**			mem_delete_tree(table_ptr[i].table_ptr);
**		} else {
**			mem_delete_leaf(table_ptr[i]);
**		}
**	}
**	mem_free_frame(table_ptr);
** }
*/
#define PTE_ENTRY_SIZE	16	
mem_delete_as:
	push	p_ra
	
	ldiq	t2, (PAGESIZE/PTE_ENTRY_SIZE)		// number of nodes to check
	bic	v0, 0x3f, v0				// mask out lower bits - this is our frame (table_ptr)
		
	REPEAT
		IF v0
			ldq_p	t3, 0(v0)
			ldq_p	t4, 8(v0)      
			and	t3, PP_INTREE, t3
			IF	t3
				push	p_v0|p_t2
				mov	t4, v0
				bsr	ra, mem_delete_as
				pop	p_v0|p_t2
			ENDIF
			// FIXME: check leaf is valid and delete.. what do we have to do here??
			subq	t2, 1, t2
			addq	v0, 16, v0
			CONT	t2
		ENDIF
	ENDR

	bic	v0, PAGEMASK, v0
	bsr	ra, mem_free_frame

	pop	p_ra
	ret	(ra)
		
		
.END
