/*  Gauntlet/kernel/arm.h
 *  Created by Adam Wiggins: 22/06/1999
 *  Last Modified by Adam Wiggins: 17/10/1999 
 *  Generic ARM defines and macros
 */

#ifndef ARM_H
#define ARM_H

#include <types.h>
#include <macros.h>

/**************************
* Program Status Register *
**************************/

/* Processor Modes */

#define USR_MODE 0x10 /* User Mode */
#define FIQ_MODE 0x11 /* Fast Interupt Mode */
#define IRQ_MODE 0x12 /* Normal Interupt Mode */
#define SVC_MODE 0x13 /* Supervisor Mode */
#define ABT_MODE 0x17 /* Abort Mode */
#define UND_MODE 0x1B /* Undefined Instruction Mode */
#define SYS_MODE 0x1F /* System Mode */

/* Condition Code Flags */

#define N_CFLAG (1 << 31) /* Negative Condition Code */
#define Z_CFLAG (1 << 30) /* Zero Condition Code */
#define C_CFLAG (1 << 29) /* Carry Condition Code */
#define V_CFLAG (1 << 28) /* Overflow Condition Code */

/* Control Bits */

#define IRQ_MASK (1 << 7) /* IRQ Disable Bit */
#define FIQ_MASK (1 << 6) /* FIQ Disable Bit */

#define MODE_MASK 0x0000000F /* (5th bit is always 1) */

#ifdef LANGUAGE_C

typedef struct cpsr {
  word_t negative:1;
  word_t zero:1;
  word_t carry:1;
  word_t overflow:1;
  word_t pad:20;
  word_t irq:1;
  word_t fiq:1;
  word_t thumb:1;
  word_t mode:5;
} cpsr_t;

#endif LANGUAGE_C

/******************
* Co-Processor 15 *
******************/

/* Register 1: Control Regster */

#define CONTR_M (1 << 0)  /* MMU Enable/Disable Bit */
#define CONTR_A (1 << 1)  /* Alignment Fault Enable/Disable Bit */
#define CONTR_C (1 << 2)  /* Data Cache Enable/Disable Bit  */
#define CONTR_W (1 << 3)  /* Write Buffer Enable/Disable Bit */
#define CONTR_P (1 << 4)  /* 26-bit/32-bit Exception Handlers */
#define CONTR_D (1 << 5)  /* 26-bit/32-bit Data Address Range */
#define CONTR_L (1 << 6)  /* Implimentation Defined */
#define CONTR_B (1 << 7)  /* Big/Little Endian Bit */
#define CONTR_S (1 << 8)  /* System Protections Bit */
#define CONTR_R (1 << 9)  /* ROM protection Bit */
#define CONTR_F (1 << 10) /* Implimentation Defined */
#define CONTR_Z (1 << 11) /* Implimentation Defined */
#define CONTR_I (1 << 12) /* Instruction Cache Enable/Disable Bit */
#define CONTR_X (1 << 13) /* Virtual Interrupt Vector Adjust */ 

#ifdef LANGUAGE_C

typedef union cntr_reg {
  word_t init;
  struct flags {
    word_t pad1:18;
    word_t x:1;
    word_t i:1;
    word_t z:1;
    word_t f:1;
    word_t r:1;
    word_t s:1;
    word_t b:1;
    word_t l:1;
    word_t d:1;
    word_t p:1;
    word_t w:1;
    word_t c:1;
    word_t a:1;
    word_t m:1;
  } flag;
} cntr_reg_t;

#endif LANGUAGE_C

/* Register 5: Fault Status Register */

#define TRANS_PAGE    0b0111
#define TRANS_SECTION 0b0101
#define PERMS_PAGE    0b1111
#define PERMS_SECTION 0b1101

#define STATUS_MASK 0x0000000F

/********* 
* Macros * 
*********/

/* Move CP15 ID register to ARM register */
#define RdCP15_ID(reg) \
	mrc	p15, 0, reg, c0, c0, 0

/* Move CP15 Control register to ARM register */
#define RdCP15_Control(reg) \
	mrc	p15, 0, reg, c1, c0, 0

/* Move ARM register to CP15 Control register */
#define WtCP15_Control(reg) \
	mcr	p15, 0, reg, c1, c0, 0

/* Move CP15 Page/Translation Table Base register to ARM register */
#define RdCP15_PTBase(reg) \
	mrc	p15, 0, reg, c2, c0, 0

/* Move ARM register to CP15 Page/Translation Table Base register */
#define WtCP15_PTBase(reg) \
	mcr	p15, 0, reg, c2, c0, 0

/* Move CP15 Domain Access Control register to ARM register */
#define RdCP15_DomAccess(reg) \
	mrc	p15, 0, reg, c3, c0, 0

/* Move ARM register to CP15 Domain Access Control register */
#define WtCP15_DomAccess(reg) \
	mcr	p15, 0, reg, c3, c0, 0

/* Move CP15 Fault Status register to ARM register */
#define RdCP15_FaultStat(reg) \
	mrc	p15, 0, reg, c5, c0, 0

/* Move ARM register to CP15 Fault Status register */
#define WtCP15_FaultStat(reg) \
	mcr	p15, 0, reg, c5, c0, 0

/* Move CP15 Fault Address register to ARM register */
#define RdCP15_FaultAddr(reg) \
	mrc	p15, 0, reg, c6, c0, 0

/* Move ARM register to CP15 Fault Address register */
#define WtCP15_FaultAddr(reg) \
	mcr	p15, 0, reg, c6, c0, 0

/* CP15 Flush ICache & DCache */
#define Flush_IDCache \
        mcr     p15, 0, r0, c7, c7, 0

/* CP15 Flush Instruction Cache */
#define Flush_ICache \
        mcr     p15, 0, r0, c7, c5, 0

/* CP15 Flush Data Cache */
#define Flush_DCache \
        mcr     p15, 0, r0, c7, c6, 0

/* CP15 Flush Data Cache Entry, virtual address in ARM register */
#define Flush_DCacheEnt(reg) \
        mcr     p15, 0, reg, c7, c6, 1

/* CP15 Clean Data Cache Entry, virtual address in ARM register */
#define Clean_DCacheEnt(reg) \
        mcr     p15, 0, reg, c7, c10, 1

/* CP15 Clean & Flush Data Cache Entry, virtual address in ARM register */
#define CleanFlush_DCacheEnt(reg) \
        mcr     p15, 0, reg, c7, c14, 1

/* CP15 Drain Write Buffer */
#define Drain_WriteBuffer \
        mcr     p15, 0, r0, c7, c10, 4

/* CP15 Flush Instruction TLB & Data TLB */
#define Flush_ID_TLB \
        mcr     p15, 0, r0, c8, c7, 0

/* CP15 Flush Instruction TLB */
#define Flush_I_TLB \
        mcr     p15, 0, r0, c8, c5, 0
 
/* CP15 Flush Data TLB */
#define Flush_D_TLB \
        mcr     p15, 0, r0, c8, c6, 0

/* CP15 Flush Data TLB entry, virtual address in ARM register */
#define Flush_D_TLB_Ent(reg) \
        mcr     p15, 0, reg, c8, c6, 1

/* The rest of CP15 macro's defined in Processor specific header files */

#ifdef LANGUAGE_C

/* Load Kernel Misc Data Pointer */
#define load_KData()            \
        do {                  \
        __asm__ __volatile__( \
        "LOAD_KDATA(kdp)"     \
         : : );               \
        } while (0)

/* Move variable to PC */
#define load_PC(x)            \
        do {                  \
        __asm__ __volatile__( \
        "mov    pc, %0"       \
         : : "r" (x));        \
        } while (0)

/* Move CP15 Control register to ARM register */
#define readCP15_control(x)            \
        do {                           \
        __asm__ __volatile__(          \
	"mrc	p15, 0, %0, c1, c0, 0" \
         : : "r" (x));                 \
        } while (0)

/* Move ARM register to CP15 Control register */
#define writeCP15_control(x)           \
        do {                           \
        __asm__ __volatile__(          \
	"mcr	p15, 0, %0, c1, c0, 0" \
         : : "r" (x));                 \
        } while (0)

/* Move CP15 Page/Translation Table Base register to ARM register */
#define readCP15_ptBase(x)             \
        do {                           \
        __asm__ __volatile__(          \
	"mrc	p15, 0, %0, c2, c0, 0" \
         : : "r" (x));                 \
        } while (0)

/* Move ARM register to CP15 Page/Translation Table Base register */
#define writeCP15_ptBase(x)            \
        do {                           \
        __asm__ __volatile__(          \
	"mcr	p15, 0, %0, c2, c0, 0" \
         : : "r" (x));                 \
        } while (0)

/* Move CP15 Domain Access Control register to ARM register */
#define readCP15_domAccess(x)          \
        do {                           \
        __asm__ __volatile__(          \
	"mrc	p15, 0, %0, c3, c0, 0" \
         : : "r" (x));                 \
        } while (0)

/* Move ARM register to CP15 Domain Access Control register */
#define writeCP15_domAccess(x)         \
        do {                           \
        __asm__ __volatile__(          \
	"mcr	p15, 0, %0, c3, c0, 0" \
         : : "r" (x));                 \
        } while (0)

#endif LANGUAGE_C


#endif ARM_H
