/*  Gauntlet/kernel/debug.c
 *  Created by Adam Wiggins: 15/08/1999
 *  Last Modified by Adam Wiggins: 09/11/1999
 *  General Debugging code for Gauntlet
 */

#include <debug.h>

/* Dump given pagetable (virtual addresses) */
void 
dumpPT(fld_t *page_dir)
{
  int i, j;
  sld_t *page_table;

  printf("\nLevel 1 Page Table Dump at %x\n", (int)page_dir);
  for(i = 0; i < FLD_MAX; i++) 
    switch (page_dir[i].invalid.type) {
      case FLD_SECTION:
        printf("Vaddr %x Section, Dom %x, ", fldToVaddr(i), 
               page_dir[i].section.domain);
        switch (page_dir[i].section.ap) {
          case AP_RESTRICTED:
            printf("AP_RESTRICTED, ");
            break;
          case AP_NOACCESS:
            printf("AP_NOACCESS, ");
            break;
          case AP_READACCESS:
            printf("AP_READACCESS, ");
            break;
          case AP_ALLACCESS:
            printf("AP_ALLACCESS, ");
            break;
        }
        printf("Paddr %x, entry %x\n", sbaToPaddr(page_dir[i].section.sba),
               page_dir[i].init);
        break;
      case FLD_FINE:
        printf("Fine page table?\n");
        break;
      case FLD_COARSE:
        printf("Vaddr %x Coarse Page, Dom %x, entry %x\n", fldToVaddr(i), 
               page_dir[i].coarse.domain, page_dir[i].init);
        page_table = 
          (sld_t *)paddrToVaddr0(cptbaToPaddr(page_dir[i].coarse.cptba));
        //page_table = (sld_t *)cptbaToPaddr(page_dir[i].coarse.cptba);
        printf("\tLevel 2 Coarse Page Table Dump at %x\n", (int)page_table);
        for(j = 0; j < CSLD_MAX; j++)
          switch (page_table[j].invalid.type) {
            case SLD_SMALL:
              printf("\tVaddr %x Small, ", (fldToVaddr(i) | csldToVaddr(j)));
              switch (page_table[j].small.ap0) {
                case AP_RESTRICTED:
                  printf("AP_RESTRICTED, ");
                  break;
                case AP_NOACCESS:
                  printf("AP_NOACCESS, ");
                  break;
                case AP_READACCESS:
                  printf("AP_READACCESS, ");
                  break;
                case AP_ALLACCESS:
                  printf("AP_ALLACCESS, ");
                  break;
              }
              printf("Paddr %x, entry %x\n", 
                     spbaToPaddr(page_table[j].small.spba),
                     page_table[j].init);
              break;
            case SLD_LARGE:
              printf("\tVaddr %x Large, ", 
                     (fldToVaddr(i) | csldToVaddr(j)));
              switch (page_table[j].large.ap0) {
                case AP_RESTRICTED:
                  printf("AP_RESTRICTED, ");
                  break;
                case AP_NOACCESS:
                  printf("AP_NOACCESS, ");
                  break;
                case AP_READACCESS:
                  printf("AP_READACCESS, ");
                  break;
                case AP_ALLACCESS:
                  printf("AP_ALLACCESS, ");
                  break;
              }
              printf("Paddr %x, entry %x\n", 
                     lpbaToPaddr(page_table[j].large.lpba),
                     page_table[j].init);
              break;
          }
        break;
    }
  puts("\n");
}

/* Dump the Task Table */
void
dumpTT(void)
{
  int i;

  printf("Dumping Task Table at %x\n", task_table);
  for(i = 0; i < MAX_TASKS; i++)
    if(task_table[i].active) {
      printf("\tTask %x is active\n", i);
      printf("\trefcount is %x\n", task_table[i].refcount);
      printf("\tdomain is %x\n", task_table[i].domain);
      printf("\tpid is %x\n", task_table[i].pid);
      printf("\tchief is %x\n", task_table[i].chief);
      printf("\tdepth is %x\n", task_table[i].depth);
    }
}

/* Dump the Kernel Data */
void
dumpKD(void)
{
  int i;

  printf("Dumping Kernel Misc Data at %x\n", kernel_data);
  printf("\tClock is %x\n", kernel_data->clock);
  printf("\tmemsize is %x\n", kernel_data->memsize);
  printf("\tfreelist_16kb is %x\n", kernel_data->freelist_16kb);
  printf("\tfreelist_4kb is %x\n", kernel_data->freelist_4kb);
  printf("\tfreelist_1kb is %x\n", kernel_data->freelist_1kb);
  printf("\tCurrent stack is %x\n", kernel_data->current_stack);
  printf("\tpresent list is %x\n", kernel_data->present_list);
  for(i = 0; i < MAX_PRIORITY; i++)
    if(kernel_data->ready_queue[i])
      printf("\tready_queue %x is non-empty starting at %x\n", 
             i, kernel_data->ready_queue[i]);
  printf("Kernel Misc Data Dump Done\n");
}

/* Dump TCB */
void 
dumpTCB(tcb_t *tcb)
{
  word_t *stack = tcb->stack_pointer;

  printf("Dumping TCB at %x\n", tcb);
  if(!(tcb->coarse_state == NULL)) {
    printf("\ttask page_dir physical address is %x\n", tcb->page_dir);
    printf("\ttask mcp is %x\n", tcb->mcp);
    printf("\tthread priority is %x\n", tcb->priority);
    printf("\tthread's coarse state is %x\n", tcb->coarse_state);
    printf("\tthread's fine state is %x\n", tcb->fine_state);
    printf("\tthread id is %x\n", tcb->myself_tid);
    printf("\tpager id is %x\n", tcb->pager_tid);
    printf("\texcpt id is %x\n", tcb->excpt_tid);
    printf("\tstack pointer is %x\n", tcb->stack_pointer);
    printf("\tIPC timeouts is %x\n", tcb->timeouts);
    printf("\tIPC partner tcb is %x\n", tcb->partner);
    printf("\tSend queue start tcb is %x\n", tcb->send_start);
    printf("\tSend queue end tcb is %x\n", tcb->send_end);
    printf("\tSend queue next tcb is %x\n", tcb->send_next);
    printf("\tSend queue prev tcb is %x\n", tcb->send_prev);
    printf("\tNext present tcb is %x\n", tcb->present_next);
    printf("\tPrev present tcb is %x\n", tcb->present_prev);
    printf("\tNext tcb at this priority is %x\n", tcb->ready_next);
    printf("\tPrev tcb at this priority is %x\n", tcb->ready_prev);
    printf("\tMagic is %x\n", tcb->magic);
    printf("\tDump of TCB stack from %x\n",tcb->stack_pointer);
    while(stack != (word_t *)(tcb + 1)) {
      printf("\t\t Stack at %x contains %x\n", stack, *stack);
      stack++;
    }
  } else
    printf("\tTCB not active\n");
}

int 
puts(char *string)
{
  int i;

  for(i = 0; string[i] != '\0'; i++) {
    putc(string[i]);
    if(string[i] == '\n')
      putc('\r');
  }

  return i;
}

int 
gets(char *string)
{
  int i;

  for(i = 0; (string[i] = getc()) != '\0'; i++) {
    putc(string[i]);
    if(string[i] == '\n') {
      string[i + 1] = '\0';
      break;
    }
  }

  return i;
}

int 
gethex(char *string)
{
  int i, digit, hex = 0;

  for(i = 0; (string[i] != '\0') && (i < 8); i++) {
    if(string[i] == '\n')
      break;
    if((digit = char2hex(string[i])) == -1) {
      hex = -1;
      break;
    }
    hex <<= 4;
    hex += digit;
  }

  return hex;
}

int 
char2hex(char chr)
{
  if((chr >= '0') && (chr <= '9'))
    return (chr - '0');
  else if((chr >= 'a') && (chr <= 'f'))
    return (chr - 'a' + 10);
  else if((chr >= 'A') && (chr <= 'F'))
    return (chr - 'A' + 10);
  else
    return -1;
}

static const byte_t hexchars[] = "0123456789ABCDEF";


void
print_hex(word_t val)
{
  int i;        /* becomes negative */
  for (i=28; i >= 0; i -= 4)
    putc(hexchars[(val >> i) & 0xF]);
};

void
print_dec(word_t val)
{
  /*  char buff[16];
  word_t i = 14;

  buff[15] = '\0';
  do
  {
      buff[i] = (val % 10) + '0';
      val = val / 10;
      i--;
  } while(val);

  
 puts(&buff[i+1]);
  */
};

int
printf(char* format, ...)
{
    int ret = 1;
    int i = 0;
    
#define arg(x) (((word_t*) &format)[(x)+1])

    /* sanity check */
    if (format == NULL)
      return 0;

    while (*format)
      {
	switch (*(format))
	  {
	  case '%':
	    switch (*(++format))
	      {
	      case 'c':
		putc(arg(i));
		break;
              case 'd':
		print_dec(arg(i));
		break;
	      case 'x':
		print_hex((word_t)arg(i));
		break;
	      case 's':
		puts((char*) arg(i));
		break;
	      default:
		puts("?");
		break;
	      };
	    i++;
	    break;
	  default:
	    putc(*format);
	    if(*format == '\n')
	      putc('\r');
	    break;
	  };
	format++;
      };
    return ret;
};
