/*  Gauntlet/kernel/init.c
 *  Created by Adam Wiggins: 22/06/1999
 *  Last Modified by Adam Wiggins: 09/11/1999
 *  Code to initialise and startup Gauntlet
 */

#include <init.h>

/* main(): Starts up L4 with Sigma0 task.
 * Pre:    system initialised to enter C.
 * Post:   Gauntlet and Sigma zero initialised, Simga0 starts executing 
 *         at its entry point. */
void 
main(void)
{
  int i;

/* Initialise/enable Memory & MMU */
  initMem();
/* Initialise scheduling */
  initSchedule();
/* Initialise Task Table */
  for(i = 0; i < MAX_TASKS; i++)
    task_table[i].active = FALSE;
/* Initialise Sigma0 */
  initSigma0();
/* Start L4 running */
  freeInit();
} /* main() */

/* initMem(): Initialises and enables kernel virtual memory.
 * Pre:       MMU disabled.
 * Post:      Kernel running in virtual memory. */
void 
initMem(void)
{
  cntr_reg_t control;
  dom_reg_t domain;

  buildBasePT();                     /* Build the kernel pagetable */
  writeCP15_ptBase(cache_page_dir_p);/* Init pagetable base register */
  domain.init = 0;                   /* Init all domains to fault */
  domain.domain.kernel = DAT_CLIENT; /* Kernel Domain is set to client */
  domain.domain.user01 = DAT_CLIENT; /* User01 Domain is set to client */
  writeCP15_domAccess(domain);       /* Set domain access control register */
  control.init = CNTRL_FLAGS;        /* Initialise control flags */
  writeCP15_control(control);        /* Set System Control Register */
  initKalloc();                      /* Init kernel memory allocator */
} /* initMem() */

/* buildBasePT(): Builds the Kernel Page Table (Caching Page Directory).
 * Pre:           MMU disabled.
 * Post:          Caching Page Directory initialised (containning only kernel
 *                mappings). */
void
buildBasePT(void)
{
  fld_section_t *section;
  int i;

/* Zero out 1st Level Array */
  for(i = 0; i < FLD_MAX; i++)
    cache_page_dir_p[i].init = 0;
/* Map in Kernel INIT Segment (temp 1-1 mapping, 1M section no caching) */
  section = (fld_section_t *)cache_page_dir_p + 
    vaddrToFLD(INIT_BASE);
  section->type = FLD_SECTION;                /* 1st Level Section Entry */
  section->ap = AP_NOACCESS;                  /* Kernel Access only */
  section->sba = paddrToSBA(INIT_BASE);       /* Address of 1MB Frame */
/* Map in Kernel */
  mapKCDA();       /* Map in Kernel Code/Data Area */
  mapPlatformIO(); /* Map in Platform I/O Registers */
  mapKDebug();     /* Map in Kernel Debugger Area */
  mapContRAM();    /* Map in RAM as a Contiguous Block */
} /* buildBasePT() */

/* mapKCDA(): Adds a mapping for the Kernel Code/Data Area into the Caching
 *            Kernel Pagetable(64KB large page Cached/Buffered). 
 * Pre:       Caching Page Directory initialised.
 * Post:      Kernel Code/Data Area mappings added to Caching Page Directory. */
void 
mapKCDA(void)
{
  sld_t *page_table = (sld_t *)(KCDA_PBASE + KCDA_DATA + 0xC00); // Maybe move
  sld_largepage_t *large_page;
  fld_coarse_t *coarse;
  int i;

/* Add 1st Level Entry */
  coarse = (fld_coarse_t *)cache_page_dir_p + vaddrToFLD(KCDA_VBASE);
  coarse->type = FLD_COARSE;
  coarse->cptba = paddrToCPTBA(page_table);
/* Zero out 2st Level Array */
  for(i = 0; i < CSLD_MAX; i++)
    page_table[i].init = 0;
/* Add 2nd Level Entry */
  large_page = (sld_largepage_t *)page_table + vaddrToCSLD(KCDA_VBASE);
  large_page->type = SLD_LARGE;
  large_page->buf = KERNEL_BUFFERABLE;
  large_page->cache = KERNEL_CACHABLE;
  large_page->ap0 = AP_NOACCESS;
  large_page->ap1 = AP_NOACCESS;
  large_page->ap2 = AP_NOACCESS;
  large_page->ap3 = AP_NOACCESS;
  large_page->lpba = paddrToLPBA(KCDA_PBASE);
  /* Copy Entry (Required for Large Pages) */
  for(i = 1; i < LRG_CENTS; i++) 
    *(large_page + i) = *large_page;
} /* mapKCDA() */

/* initSchedule(): Initialises L4 scheduling data structures.
 * Pre:            Kernel virtual memory initialised and enabled.
 * Post:           L4 Scheduler initialised.
 * Status:         Wait till we impliment scheduling, what else needed?
 * Priority:       Important but not immediate. */
void 
initSchedule(void)
{
  int i;

/* Initialise Scheduling Kernel Data Structures */
  kernel_data->clock = 0;               /* System boot time */
  kernel_data->present_list = NULL;     /* Empty present list */
  for(i = 0; i < MAX_PRIORITY; i++)     /* Empty ready thread queues */
    kernel_data->ready_queue[i] = NULL;
  initOSTimers();                       /* Initialise Kernel OS Timers */
} /* initSchedule() */

/* initSigma0(): Initialises and schedules Sigma0 task.
 * Pre:          L4 initialised and ready for scheduling.
 * Post:         Sigma0 task in ready queue, scheduling disabled.
 * Status:       Pretty much done.
 * NOTE:         When fast address space switches are implimented Sigma0 must
 *               be given it's own page direcotry and this added to the page
 *               directory virtual array. */
void 
initSigma0(void)
{ 
  tcb_t *sigma0_tcb = tcbs + tidToTCB(SIGMA0_TID);
  task_table_t *sigma0_tt = task_table + tidToTASK(SIGMA0_TID);

/* task_new() Simga0 */
  initSigma0TCB();
  /* Set up Task Table info for Sigma0 */
  sigma0_tt->refcount = 0;
  sigma0_tt->domain = USER01_DOMAIN; /* First domain */
  sigma0_tt->pid = 0x0;              /* Currently unused */
  sigma0_tt->chief = 
    tidToTASK(KERNEL_TASK);          /* Kernel is the Chief (reserved task) */
  sigma0_tt->depth = 0;              /* Root of Clan's & Chief's */
  sigma0_tt->active = TRUE;          /* Mask Sigma0 as active */
  /* Set up Kernel Misc Data info for Sigma0 */
  kernel_data->current_stack = (word_t *)(sigma0_tcb + 1);
  kernel_data->present_list = sigma0_tcb;
  kernel_data->ready_queue[0] = sigma0_tcb;
  dumpKD();
/* Build Simga0 Page Table */
  // FILL ME - Add the sigma0 page directory to virtual page directory array
  mapSigma0();
} /* initSigma0() */

/* initSigma0TCB(): Allocates frame for Sigma0 lthread0 TCB, initialises TCB's
                    in this frame. 
   Pre:             Running in Virtual Memory.
   Post:            Sigma0 lthread0 initialised. Remainning TCB's set to 
                    invalid.
   Status:          Pretty much done. */
void initSigma0TCB(void)
{
  int i;
  tcb_t *sigma0_tcb = tcbs + tidToTCB(SIGMA0_TID);
  freelist_t *tcb;
  sld_t *page_table_p, *page_table;
  fld_coarse_t *page;
  sld_smallpage_t *small_page;

/* Map in TCB Frame */
  /* Allocate TCB Frame */
  tcb = kernel_data->freelist_4kb;
  kernel_data->freelist_4kb = kernel_data->freelist_4kb->next;
  /* Allocate 2nd level Array */
  page_table = (sld_t *)kernel_data->freelist_1kb;
  kernel_data->freelist_1kb = kernel_data->freelist_1kb->next;
  page_table_p = (sld_t *)vaddrToPaddr0(page_table);
  /* Add 1st Level Entry */
  page = (fld_coarse_t *)cache_page_dir + vaddrToFLD(sigma0_tcb);
  page->type = FLD_COARSE;
  page->cptba = paddrToCPTBA(page_table_p);
  /* Zero out 2nd Level Array */
  for(i = 0; i < CSLD_MAX; i++)
    page_table[i].init = 0;
  /* Add 2nd Level Entry */
  small_page = (sld_smallpage_t *)page_table + vaddrToCSLD(sigma0_tcb);
  small_page->type = SLD_SMALL;
  small_page->buf = KERNEL_BUFFERABLE;
  small_page->cache = KERNEL_CACHABLE;
  small_page->ap0 = AP_NOACCESS;
  small_page->ap1 = AP_NOACCESS;
  small_page->ap2 = AP_NOACCESS;
  small_page->ap3 = AP_NOACCESS;
  small_page->spba = paddrToSPBA(vaddrToPaddr0(tcb));
/* Initilise Sigma0 lthread0 TCB */
  sigma0_tcb->page_dir = cache_page_dir_p;
  sigma0_tcb->mcp = 0x0;
  sigma0_tcb->priority = 0x0;
  sigma0_tcb->coarse_state = CS_THREAD_ACTIVE;
  sigma0_tcb->fine_state = FS_READY;
  sigma0_tcb->myself_tid.init = SIGMA0_TID;
  sigma0_tcb->pager_tid.init = KERNEL_TASK;
  sigma0_tcb->excpt_tid.init = KERNEL_TASK;
  sigma0_tcb->stack_pointer = (word_t *)(sigma0_tcb + 1);
  sigma0_tcb->timeouts = 0x0;
  sigma0_tcb->partner = NULL;
  sigma0_tcb->send_start = NULL;
  sigma0_tcb->send_end = NULL;
  sigma0_tcb->send_next = NULL;
  sigma0_tcb->send_prev = NULL;
  sigma0_tcb->present_next = sigma0_tcb;
  sigma0_tcb->present_prev = sigma0_tcb;
  sigma0_tcb->ready_next = sigma0_tcb;
  sigma0_tcb->ready_prev = sigma0_tcb;
  sigma0_tcb->magic = 0x13579BDF;
/* Initialise Sigma0 Kernel Stack */
  *(--sigma0_tcb->stack_pointer) = SIGMA0_INIT_PC;
  *(--sigma0_tcb->stack_pointer) = SIGMA0_INIT_SP;
  *(--sigma0_tcb->stack_pointer) = (USR_MODE | FIQ_MASK | IRQ_MASK);
  *(--sigma0_tcb->stack_pointer) = (word_t)fast_syscall_return;
/* Invalidate Remainning TCB's in Frame */                          
  for(i = 1; i < (PAGE_SIZE / TCB_SIZE); i++)
    (sigma0_tcb + i)->coarse_state = CS_UNUSED_TCB;
/* TEMP TESTING */
  dumpTCB(sigma0_tcb);
  dumpTCB(sigma0_tcb + 1);
} /* initSigma0TCB */

/* initKalloc(): Initialises Kernel SLAB allocator.
 * Pre:          Running in Virtual Memory.
 * Post:         Kernel freelists initialised.
 * Status:       Done, Bit Messy still. */
void 
initKalloc()
{
  freelist_t *next;

  kernel_data->memsize = memSize();
/* Build 1KB frame freelist */
  /* Add spare frames from kernel code/data area */
  // FILL ME!
  /* Allocate frames from start of free ram area */
  next = (freelist_t *)(0xF0010000); // Dodgy, FIXME! 
  kernel_data->freelist_1kb = next;
  while(next < (freelist_t *)MAX_1KB_FRAME) {
    next->next = next + CSLD_MAX;
    next = next->next;
  }
  next->next = NULL;
/* Build 4KB frame freelist */
  next += CSLD_MAX;
  kernel_data->freelist_4kb = next;
  while(next < (freelist_t *)MAX_4KB_FRAME) {
    next->next = next + FSLD_MAX;
    next = next->next;
  }
  next->next = NULL;
/* Build 16KB frame freelist */
  next += FSLD_MAX;
  kernel_data->freelist_16kb = next;
  while(next < (freelist_t *)MAX_16KB_FRAME) {
    next->next = next + FLD_MAX;
    next = next->next;
  }
  next->next = NULL;
} /* initKalloc() */

/* free_init: Removes initialisation code and enables L4 scheduling.
 * Pre:       L4 & Sigma0 initialised.
 * Post:      L4 running.
 * Status     Think its done, needs testing. */
void 
freeInit(void)
{
  tcb_t *sigma0_tcb = tcbs + tidToTCB(SIGMA0_TID);
  fld_section_t *section;

/* FIXME! Remove INIT segment mapping */
  section = (fld_section_t *)cache_page_dir_p + 
    vaddrToFLD(INIT_BASE);
  section->type = FLD_INVALID;
/* Context switch to Simga0 */
  //  load_KData();
  load_PC(*(sigma0_tcb->stack_pointer++)); /* Jump to syscall return handler */
} /* freeInit() */
