/*  Gauntlet/kernel/ipc.S
 *  Created by Adam Wiggins: 16/10/1999
 *  Last Modified by Adam Wiggins: 27/10/1999
 *  Generic ARM IPC code
 */

#include <asm.h>
#include <ipc.h>
#include <tcb.h>
#include <types.h>
#include <kernel.h>
#include <macros.h>

/* no_partner: Return Error code for non existant IPC partner.
 * Pre:        None.
 * Post:       r3 = IPC_INVALID_PARTNER,
 *             return from syscall. */
EXCP(no_partner)
	mov	r3, #IPC_INVALID_PARTNER	@ Load invalid partner result
	b	fast_syscall_return		@ Return from syscall
/* end no_partner */

/* send_timeout: Return Error code for timeout of the send phase of IPC.
 * Pre:          None.
 * Post:         r3 = IPC_SEND_TIMEDOUT,
 *               return from syscall. */
EXCP(send_timedout)
	mov	r3, #IPC_SEND_TIMEDOUT		@ Load send timeout result
	b	fast_syscall_return		@ Return from syscall
/* end send_timeout */

/* receive_timeout: Return Error code for timeout of the receive phase of IPC.
 * Pre:             Send phase (if any) completed.
 * Post:            r3 = IPC_RECEIVE_TIMEDOUT,
 *                  return from syscall. */
EXCP(receive_timedout)
	mov	r3, #IPC_RECEIVE_TIMEDOUT	@ Load receive timeout result
	b	fast_syscall_return		@ Return from syscall
/* end receive_timeout */

/* kernel_sender: Sender TCB is a kernel task thread.
 * Pre:           ?
 * Post:          ?
 * Status:        Not done, this is for handling interrupts essentially. */
EXCP(kernel_sender)
	b	fast_syscall_return
/* end kernel_sender */
	
/* stack_after_send: Stack sender to wait for receiver after send operation.
 * Pre:              ?
 * Post:             ?
 * Status:           Not started yet, skip recieve.
 * Priority:         Look at after send/receiver only operations completed. */
EXCP(stack_after_send)
	b	finish_send	@ Return to finish send  
/* end stack_after_send */

/* stack_after_receive: Stack sender, wait for receiver after receive operation.
 * Pre:              ?
 * Post:             ?
 * Status:           Not started yet, skip receive.
 * Priority:         Bother with once send/receive only operations done. */
EXCP(stack_after_receive)
	b	finish_receive	@ Return to finish receive
/* end stack_after_receive */

/* receiver_not_ready: Stack sender and set to polling on receiver.
 * Pre:                r0 = Current Thread's Kernel Stack Pointer,
 *                     r1 = Partner (receiver) Thread's Control Blocks,
 *                     r2 = Send Descriptor,
 *                     r3 = Reciever Descriptor,
 *                     r4 = Timeouts, 
 *                     r5 - r12 = Short Message,
 *                     r14 = Current (sender) Thread's Control Block.
 * Post:               Sender thread stacked and polling receiver,
 *                     Added to receivers send queue,
 *                     r13 = Kernel Misc Data Pointer.
 * Registers:          r13 = Temp.
 * Status:             Untested,
 *                     Lacks real timeouts (inf/zero only). */
EXCP(receiver_not_ready)
	tst	r4, #SEND_TIMEOUT_EXP		@ Test send timeout exp
	beq	send_timedout			@ If zero, send_timedout
	str	r4, [r14, #T_TIMEOUTS]		@ Store timeouts in callers TCB
	mov	r13, #FS_POLLING		@ Set threads state to polling
	str	r13, [r14, #T_FINE_STATE]	@ Store the new fine state
/* Add caller thread to receivers send queue */
	ldr	r13, [r1, #T_SEND_END]		@ Load receivers send queue last
/* R */	str	r1, [r14, #T_PARTNER]		@ Store callers partner
	teq	r13, #NULL			@ Test if send queue empty
	bne	send_queue_add			@ If not empty, send_queue_add
/* Start new send queue */
	str	r14, [r1, #T_SEND_START]	@ Set caller head of send queue
	str	r14, [r1, #T_SEND_END]		@ Set caller end of send queue
	str	r13, [r14, #T_SEND_NEXT]	@ Set callers send next to NULL
	str	r13, [r14, #T_SEND_PREV]	@ Set callers send prev to NULL
	b	stack_sender			@ Return to stack_sender 
send_queue_add:					@ Add to existing send queue
	str	r14, [r13, #T_SEND_NEXT]	@ Set caller as lasts send next
	str	r13, [r14, #T_SEND_PREV]	@ Set last as callers send prev
	str	r14, [r1, #T_SEND_END]		@ Set caller as send queue last
	mov	r13, #NULL			@ Load NULL into Temp
	str	r13, [r14, #T_SEND_NEXT]	@ Set callers send prev to NULL
stack_sender:					@ Stack IPC send state
	stm	r0!, {r2, r3, r5 - r12}		@ Stack descriptors and message
	str	r0, [r14, #T_STACK_POINTER]	@ Store callers stack pointer
	mov	r14, r7				@ Move callers tcb
	LOAD_KDATA(r13)				@ Load Kernel Misc Data Pointer
	b	dispatch_next			@ Dispatch next ready thread
/* end receiver_not_ready */

/* ipc_receive_return: Return to thread from IPC receive phase.
 * Pre:                r0 = Caller Thread's Kernel Stack Pointer.
 * Post:               User sp, CPSR restored.
 * Registers:          r14 = Temp.
 * Status:             Untested. */
EXCP(ipc_receive_return)
	ldr	r3, [r0], #8	@ Unstack IPC CC result
        POP(r0, r14)		@ Unstack SPSR into Temp
/* S */ msr     spsr, r14       @ LOAD SPSR from Temp
        ldmfd   r0, {sp, pc}^   @ Restore User sp, CPSR and pc
/* end ipc_receive_return */

/* sender_not_ready: Stack receiver and set as waiting on sender.
 * Pre:              r0 = Current Thread's Kernel Stack Pointer,
 *                   r1 = IPC Partners TCB (NULL if open wait),
 *                   r2 = Current Thread's Control Block,
 *                   r3 = Receive Descriptor,
 *                   r4 = Timeouts,
 *                   r13 = Kernel Misc Data Pointer.
 * Post:             Receiver thread stack and waiting on sender.
 * Registers:        r14 = Temp.
 * Status:           Untexted,
 *                   Lacks real timeouts (inf/zero only). */
EXCP(sender_not_ready)
        tst     r4, #RECEIVE_TIMEOUT_EXP        @ Test receive timeout exp
        beq     receive_timedout                @ If zero, receive_timedout
        str     r4, [r2, #T_TIMEOUTS]           @ Store timeouts in callers TCB
        mov     r14, #FS_WAITING                @ Set threads state to waiting
        strb    r14, [r2, #T_FINE_STATE]        @ Store the new fine state
        str     r1, [r2, #T_PARTNER]            @ Store callers partner
/* Stack receiver state */
        PUSH(r0, r3)                            @ Stack receive descriptor
	mov	r14, #IPC_OK			@ Load IPC_OK result
	PUSH(r0, r14)				@ Stack IPC CC result
        adr     r14, ipc_receive_return         @ Load return procedure addr
        PUSH(r0, r14)                           @ Stack return procedure addr
        str     r0, [r2, #T_STACK_POINTER]      @ Store callers stack pointer
        b       dispatch_next                   @ Dispatch next ready thread
/* end sender_not_ready */

/* open_receive: Open receive so check for ready sender.
 * Pre:          r3 = Receive Descriptor.
 * Post:         r1 = Destination Thread's Control Block, OR
 *               receiver stacked and set to waiting state & r1 = NULL,
 *               r2 = Caller (receiver) Thread's Control Block.
 * Status        Untested. */
EXCP(open_receive)
	KSP2TCB(r2, r0)				@ Load callers TCB into stcb
	ldr	r1, [r2, #T_SEND_START]		@ Load head of send_queue
/* R */	bic	r3, r3, #OPEN_RECEIVE		@ Clear open receive bit
	teq	r1, #NULL			@ Test for empty send_queue
	bne	do_receive			@ If non-empty, do_receive
	beq	sender_not_ready		@ Else, sender_not_ready
/* end open_receive */

/* not_short_send: Do Long IPC and/or fpage send.
 * Pre:            ?
 * Post:           ?
 * Status:         Not started yet.
 * Priority:       Once simple IPC case is done do fpage mappings, 
 *                 Long IPC is low priority. */
EXCP(not_short_send)
	// FILL ME!
	b	short_send			@ Return to short_send
/* end not_short_send */

/* not_short_receive: Do Long IPC and/or fpage receive.
 * Pre:               ?
 * Post:              ?
 * Status:            Not started yet.
 * Priority:          Same as not_short_send. */
EXCP(not_short_receive)				@ Long and/or fpage receive
	// FILL ME!
	b	short_receive			@ Return to short_receive
/* end not_short_send */
