/*  Gauntlet/kernel/kernel.h
 *  Created by Adam Wiggins: 22/06/1999
 *  Last Modified by Adam Wiggins: 06/11/1999
 *  Generic ARM Kernel structures and defines.
 */

#ifndef KERNEL_H
#define KERNEL_H

#include <tcb.h>
#include <types.h>

/*******************
* Kernel Constants *
*******************/

#define MAX_TASKS        (1 << MAX_TASKS_LOG2)
#define MAX_THREADS      (1 << MAX_THREADS_LOG2)

#define MAX_TASKS_LOG2   10 /* 1024 Tasks */
#define MAX_THREADS_LOG2 8  /* 256 Threads per Task */

#define MAX_PRIORITY     256

#define NIL_TID     0x0        /* NIL Thread */
#define INVALID_TID 0xFFFFFFFF /* INVALID Thread */
#define SIGMA0_TID  0x00040000 /* Sigma0 Task ID */
#define KERNEL_TASK 0x0003FF00 /* The Reserved Kernel Task */

#define SYSCALL_LIMIT 0x7 /* 7 Syscalls (0 - 6) */

/* Temp memory constants */

#define MAX_1KB_FRAME  0xF009FC00
#define MAX_4KB_FRAME  0xF013F000
#define MAX_16KB_FRAME 0xF01FC000

/*********************
* Virtual Memory Map *
*********************/

/* NOTE: If you edit these make sure the symbols in symbols.ld match! */

#define NULL_VBASE   0x00000000 /* 4KB, NULL Page */
#define USER_VBASE   0x00001000 /* 4GB - 256MB - 4KB, User Address Space */
#define RAM_VBASE    0xF0000000 /* 128MB, Contiguous Memory */
#define PSA_VBASE    0xF8000000 /* 64MB - 16MB, Unallocated Area */
#define PDA_VBASE    0xFB000000 /* 16MB, Page Directory Array */
#define TCBA_VBASE   0xFC000000 /* 64MB, TCB Virtual Array */ 
#define KCDA_VBASE   0xFFFF0000 /* 64KB, Kernel Code/Data Area */

/********************************
* Kernel Code/Data Area Offsets *
********************************/

#define KCDA_EXCP 0x0000 /* 4KB, Kernel EXCP Segment */
#define KCDA_TEXT 0x1000 /* 36KB, Kernel TEXT Segment */
#define KCDA_DATA 0xA000 /* 4KB, Kernel Misc Data */
#define KCDA_TT   0xB000 /* 4KB, Kernel Task Table */
#define KCDA_CPD  0xC000 /* 16KB, Caching Page Directory */

/************
* TID Table *
************/

#define TT_INACTIVE_MASK 0x00000001 /* bit 0 */
#define TT_PID_MASK      0x000000FE /* 7 bits, 7-1 */
#define TT_DOMAIN_MASK   0x00000F00 /* 4 bits, 11-8 */
#define TT_DEPTH_MASK    0x0000F000 /* 4 bits, 15-12 */
#define TT_CHIEF_MASK    0x03FF0000 /* 10 bits, 25-16 */
#define TT_REFCOUNT_MASK 0xFC000000 /* 6 bits, 31-26 */

#ifdef LANGUAGE_C

typedef struct task_table {
  word_t active:1;   /* Whether task is active or not */
  word_t pid:7;      /* PID allocated to task, currently unsed */
  word_t domain:4;   /* Domain allocated to task, 0 = no domain allocated */
  word_t depth:4;    /* Depth of this task in clan's & chief's */
  word_t chief:10;   /* Chief (Task Number) of this task */
  word_t refcount:6; /* Domain switch reference count */
} task_table_t;

#endif LANGUAGE_C

/***********************
* Kernel Data Struture *
***********************/

/* Context Info */
#define K_CURRENT_STACK 0 /* 4, Kernel Stack of Current thread, threads ID */

/* Misc Kernel data */
#define K_CLOCK 4 /* 8, number of 1 millisecond ticks since L4 booted */

/* Memory Info */
#define K_MEMSIZE       12 /* 4, Size of System Memory */
#define K_FREELIST_16KB 16 /* 4, 16KB frame freelist */
#define K_FREELIST_4KB  20 /* 4, 4KB frame freelist */
#define K_FREELIST_1KB  24 /* 4, 1KB frame freelist */

/* Scheduling Info */
#define K_PRESENT_LIST 28 /* 4, Present Thread List */
#define K_HIGHEST_PRIO 32 /* 4, Highest runnable priority + tcb prio offset */
#define K_READY_QUEUE  36 /* 4 x MAX_PRIORITY, Ready Queues */

/* End of Kernel Data */
#define K_END 1060 /* double check this */

/* Start of TID Table */
#define K_TTABLE

/* NOTE: Kernel Data CAN NOT be larger then 2048 bytes else the initial 
 *       second level coarse page tables for KCDA and I/O must be moved */

#ifdef LANGUAGE_C

typedef struct freelist {
  struct freelist *next;
} freelist_t;

typedef struct kernel_data {
  word_t     *current_stack;
  dword_t    clock;
  word_t     memsize;
  freelist_t *freelist_16kb;
  freelist_t *freelist_4kb;
  freelist_t *freelist_1kb;
  struct tcb *present_list;
  word_t     highest_prio;
  struct tcb *ready_queue[MAX_PRIORITY];
} kernel_data_t;
  
#endif LANGUAGE_C

/*********
* Macros *
*********/

/* Thread ID to Task Number */
#define tidToTASK(tid) (((word_t)(tid) & ~TID_VERSION_MASK) >> MAX_THREADS_LOG2)


#endif KERNEL_H
