/*  Gauntlet/kernel/macros.h
 *  Created by Adam Wiggins: 12/09/1999
 *  Last Modified by Adam Wiggins: 06/11/1999
 *  Macros for assembly procedures in kernel
 */

#ifndef MACROS_H
#define MACROS_H

#include <kernel.h>

/* LOAD_KDATA(reg): Load address of Kernel Data into reg.
 * Post:            reg = Kernel Misc Data Pointer. */
#define LOAD_KDATA(reg)                                                        \
	mov	reg, $0xFFFFFF00                                              ;\
	bic	reg, reg, $0x00005F00
/* end LOAD_KDATA() */

/* DISPATCH_THREAD(dest, tmp1, tmp2): Dispatch to destination thread.
 * Pre:                               dest = Destination TCB,
 *                                    tmp1 = Temp 1,
 *                                    tmp2 = Temp 2,
 *                                    r13 = Kernel Misc Data Pointer, 
 *                                    Current thread stacked (sp stored in tcb).
 * Post:                              Switch to destination thread's context and
 *                                    jump to its return procedure,
 *                                    r13 unchanged.
 * Registers:                         r0 = Current Threads Kernel Stack Pointer.
 * Status:                            On the way, untested so far. */
#define DISPATCH_THREAD(dest, tmp1, tmp2)                                      \
/* Check if task context needs switching */                                    \
	ldr	tmp1, [dest, $T_PAGE_DIR]    /* Load dest page_dir */         ;\
/* R */ RdCP15_PTBase(tmp2)                  /* Load current page_dir */      ;\
	teq	tmp2, tmp1                   /* Check if they are the same*/  ;\
	beq	255f                         /* If different, go on */        ;\
/* Switch Task Context */                                                      \
	WtCP15_PTBase(tmp1)                  /* Set new page_dir */           ;\
	FLUSH_CACHES_TLBS()                  /* Flush/Clean Caches/TLBs */    ;\
/* Switch Thread Context */                                                    \
255:	/* FILL ME - What else here? */                                        \
	ldr	r0, [dest, $T_STACK_POINTER] /* Load dest thread's SP */      ;\
	add	tmp1, dest, $TCB_SIZE        /* Set thread's empty SP */      ;\
	str	tmp1, [r13, $K_CURRENT_STACK]/* Load it into kernel Data */   ;\
/* Call Thread's Return Procedure */                                           \
	POP(r0, pc)                          /* Branch to return procedure */ ;\
/* end DISPATCH_THREAD() */

/* SYSCALL_MASK_FIQ(tmp): Set the CPSR to SVC mode with I/F bits set.
 * Pre:                   tmp = Temp.
 * Post:                  In supervisor mode,
 *                        IRQ/FIQ signals masked,
 *                        Status flags (N,V,C,Z) cleared. */
#define SYSCALL_MASK_FIQ(tmp)                                                  \
	mov	tmp, $(SVC_MODE | IRQ_MASK | FIQ_MASK) /* Load new CPSR */    ;\
	msr	cpsr, tmp                              /* Set new CPSR */
/* end SYSCALL_MASK_FIQ() */
 
/* STACK_USER_SYSCALL(): Stack the user state for a syscall.
 * Pre:                  r13 = Kernel Misc Data Pointer,
 * Post:                 r0 = Current Thread's Kernel Stack Pointer,
 *                       Syscall Stack Frame set up on callers Kernel Stack.
 *                       r13 unchanged.
 * Registers:            r14 = Temp. */
#define SYSCALL_STACK_USER()                                                   \
	ldr	r0, [r13, $K_CURRENT_STACK] /* Load Thread's SP */            ;\
/* S */	PUSH(r0, r14)                       /* Stack svc_lr */                ;\
	stmfd	r0, {sp}^                   /* Stack usr_sp */                ;\
	mrs	r14, spsr                   /* Load SPSR into Temp */         ;\
	str	r14, [r0, $-8]!             /* Stack SPSR */                  ;\
	SYSCALL_MASK_FIQ(r14)               /* Mask out FIQs */
/* end SYSCALL_STACK_USER() */

/* EXCEPTION_STACK_USER(): Stack the user state for an exception.
 * Pre:                    None.
 * Post:                   
 * Registers:              r13 = Temp. */
#define EXCEPTION_STACK_USER()                                                 \
	mov	r0, r13                    /* Move cksp to Temp */            ;\
	LOAD_KDATA(r0)                     /* Load kpd into cksp */           ;\
	ldr     r0, [r0, $K_CURRENT_STACK] /* Load Thread's SP */             ;\
/* S */	PUSH(r0, r14)                      /* Stack p_lr */                   ;\
        stmfd   r0, {r1 - r12, sp, lr}^    /* Stack User registers */         ;\
	PUSH(r0, r13)                      /* Stack r13 (was r0) */           ;\
        mrs     r14, spsr                  /* Load SPSR into r14 */           ;\
        str     r14, [r0, $-60]!           /* Stack SPSR */
/* end EXCEPTION_STACK_USER() */

/* EXCEPTION_STACK_KERNEL(): Stack the kernel state for an exception.
 * Pre:                      r0 = Current Thread's Kernel Stack,
 *                           r13 = SPSR.
 * Post:                     Kernel Exception Stack Frame set up on current 
 *                           threads kernel stack.
 * Registers:                r14 = Temp. */
#define EXCEPTION_STACK_KERNEL()                                               \
	PUSH(r0, r14)           /* Stack p_lr */                              ;\
	stmfd	r0!, {r1 - r12} /* Stack r1 to r12 */                         ;\
	PUSH(r0, r13)           /* Stack SPSR */
/* end EXCEPTION_STACK_KERNEL() */

/* PUSH(sp, reg): Push register onto the given stack pointer.
 * Pre:           sp is a valid stack pointer.
 * Post:          sp -= (-4),
 *                mem[sp] = reg,
 *                reg is unmodified. */
#define PUSH(sp, reg)                                                          \
	str	reg, [sp, $-4]!
/* end PUSH() */

/* POP(sp, reg): Pop register off the given stack pointer.
 * Pre:          sp is a valid non-empty stack pointer.
 * Post:         reg = mem[sp],
 *               sp += 4. */
#define POP(sp, reg)                                                           \
	ldr	reg, [sp], $4
/* end POP() */

/* INSERT_READY_QUEUE(tcb, tmp1, tmp2): Add 'tcb' to end of its priority ready
 *                                      queue.
 * Pre:                                 r13 = Kernel Misc Data Pointer, 
 *                                      tmp1 = Temp 1,
 *                                      tmp2 = Temp 2,
 *                                      tcb = Valid thread control block.
 * Post:                                tcb is added to the end of the
 *                                      ready_queue[prio(tcb)] (if not already 
 *                                      there).
 * Status:                              Untested. */
#define INSERT_READY_QUEUE(tcb, tmp1, tmp2)                                    \
	ldr	tmp2, [tcb, $T_READY_NEXT]  /* Load tcb's ready queue next */ ;\
/* R */	ldrb	tmp1, [tcb, $T_PRIORITY]    /* Load tcb's priority */         ;\
	teq	tmp2, tcb                   /* Test if next is itself */      ;\
	bne	255f                        /* If not, return */              ;\
/* tcb is not in a ready queue */                                              \
	ldr	tmp2, [r13, $K_HIGHEST_PRIO]/* Load highest current priority*/;\
/* R */	add	tmp1, tmp1, $K_READY_QUEUE  /* Set offset into ready queue */ ;\
	cmp	tmp2, tmp1                  /* Compare priorities */          ;\
	ldr	tmp2, [r13, tmp1]           /* Load head of ready queue */    ;\
/* R */	strlt	tmp1, [r13, $K_HIGHEST_PRIO]/* Store new highest priority */  ;\
	cmp	tmp2, $NULL                 /* Check if ready queue empty */  ;\
	bne	254f                        /* If empty, go on */             ;\
	str	tcb, [r13, tmp1]            /* Store tcb as head of queue */  ;\
	b	255f                        /* Branch to return */            ;\
254:                                        /* Non-emptry ready queue */       \
	str	tmp2, [tcb, $T_READY_NEXT]  /* set head as tcb's ready next */;\
 	ldr	tmp1, [tmp2, $T_READY_PREV] /* Load head's ready prev */      ;\
/* S */	str	tmp1, [tcb, $T_READY_PREV]  /* set head's prev as tcb's */    ;\
 	str	tcb, [tmp1, $T_READY_NEXT]  /* set tcb as prev's next */      ;\
255:                                        /* Return */
/* end INSERT_READY_QUEUE() */

/* FLUSH_CACHES_TLBS(): Flush/Clean Caches and TLBs.
 * Pre:                 r0 = Current Thread's Kernel Stack Pointer.
 * Post:                All registers untouched/restored.
 * Status:              Not done. */
#define FLUSH_CACHES_TLBS()                                                    \
/*	stmfd	r0!, {r0 - r2}            /* Stack r0 - r2 */                 ;\
/*	ldr	r0, =ZERO_PAGE_BASE       /* Load zero page mappings start */ ;\
/*	add	r1, r0, $MAIN_DCACHE_SIZE /* Add cache size */                ;\
255:                                                                          ;\
/*	ldr	r2, [r1], $32             /* clean cache line */              ;\
/*	teq	r1, r0                    /* Are we done yet? */


#endif MACROS_H
