/*  Gauntlet/kernel/mmu.h
 *  Created by Adam Wiggins: 22/06/1999
 *  Last Modified by Adam Wiggins: 14/09/1999
 *  Generic ARM MMU control defines and macros
 */

#ifndef MMU_H
#define MMU_H

#include <types.h>
#include <kernel.h>

#define PAGE_SIZE 4096

/**************************
* First-Level Descriptors *
**************************/

/* First-Level Formats */

#define FLD_INVALID 0 /* Fault Entry */ 
#define FLD_COARSE  1 /* Coarse Page Entry (Has course 2nd-level page table) */
#define FLD_SECTION 2 /* Section Entry (1MB mapping) */
#define FLD_FINE    3 /* Fine Page Entry (Has fine 2nd-level page table) */

/* First-Level Shifts */

#define FLD_DOMAIN 5  /* Domain Shift, 4bits */
#define FLD_AP     10 /* Access Permissions, 2bits */
#define FLD_CPTBA  10 /* Coarse Page Table Base Address, 22bits */
#define FLD_FPTBA  12 /* Fine Page Table Base Address, 20bits */
#define FLD_SBA    20 /* Section Base Address, 12bits */

#ifdef LANGUAGE_C

typedef struct fld_invalid {
  word_t type:2;
  word_t pad:30;
} fld_invalid_t;

typedef struct fld_coarse {
  word_t type:2;
  word_t pad2:3;
  word_t domain:4;
  word_t pad1:1;
  word_t cptba:22;
} fld_coarse_t;

typedef struct fld_section {
  word_t type:2;
  word_t buf:1;
  word_t cache:1;
  word_t pad3:1;
  word_t domain:4;
  word_t pad2:1;
  word_t ap:2;
  word_t pad1:8;
  word_t sba:12;
} fld_section_t;

typedef struct fld_fine {
  word_t type:2;
  word_t pad2:3;
  word_t domain:4;
  word_t pad1:3;
  word_t fptba:20;
} fld_fine_t;

typedef union fld {
  word_t init;
  fld_invalid_t invalid;
  fld_coarse_t coarse;
  fld_section_t section;
  fld_fine_t fine;
} fld_t;

#endif LANGUAGE_C

/**********************************
* Second-Level Descriptors *
**********************************/

/* Second-Level Formats */

#define SLD_INVALID 0 /* Fault Entry */
#define SLD_LARGE   1 /* Large Page Entry (64KB mapping) */
#define SLD_SMALL   2 /* Small Page Entry (4KB mapping) */
#define SLD_TINY    3 /* Tiny Page Entry (1KB mapping) */

/* Second-Level Shifts */

#define SLD_AP0  4  /* 1st Sub-Page Access Permissions, 2bits */ 
#define SLD_AP1  6  /* 2nd Sub-Page Access Permissions, 2bits */
#define SLD_AP2  8  /* 3rd Sub-Page Access Permissions, 2bits */
#define SLD_AP3  10 /* 4th Sub-Page Access Permissions, 2bits */
#define SLD_LPBA 16 /* Large Page Base Address, 16bits */
#define SLD_SPBA 12 /* Small Page Base Address, 20bits */
#define SLD_TPBA 10 /* Tiny Page Base Address, 22bits */

#ifdef LANGUAGE_C

typedef struct sld_invalid {
  word_t type:2;
  word_t pad:30;
} sld_invalid_t;

typedef struct sld_largepage {
  word_t type:2;
  word_t buf:1;
  word_t cache:1;
  word_t ap0:2;
  word_t ap1:2;
  word_t ap2:2;
  word_t ap3:2;
  word_t pad:4;
  word_t lpba:16;
} sld_largepage_t;

typedef struct sld_smallpage {
  word_t type:2;
  word_t buf:1;
  word_t cache:1;
  word_t ap0:2;
  word_t ap1:2;
  word_t ap2:2;
  word_t ap3:2;
  word_t spba:20;
} sld_smallpage_t;

typedef struct sld_tinypage {
  word_t type:2;
  word_t buf:1;
  word_t cache:1;
  word_t ap:2;
  word_t pad:4;
  word_t tpba:22;
} sld_tinypage_t;

typedef union sld {
  word_t init;
  sld_invalid_t invalid;
  sld_largepage_t large;
  sld_smallpage_t small;
  sld_tinypage_t tiny;
} sld_t;

#endif LANGUAGE_C

/**************
* MMU Defines *
**************/

#define FLD_SHIFT (32 - 12) /* Number of Bits to Shift to Index into First-Level array */ 
#define CSLD_SHIFT (32 - 20)  /* Number of Bits to Shift to Index into Coarse-Second-Level array */
#define FSLD_SHIFT (32 - 22) /* Number of Bits to Shift to Index into Fine-Second-Level array */
#define CSLD_MASK  (0x000000FF)
#define FLD_SIZE  (1 << 14) /* Size of First-Level Page Table in Bytes */ 
#define CSLD_SIZE (1 << 10) /* Size of Coarse-Second-Level Page Table in Bytes */
#define FSLD_SIZE (1 << 12) /* Size of Fine-Second-Level Page Table in Bytes */  
#define FLD_MAX   (FLD_SIZE >> 2) /* Number of First-Level Entries */
#define CSLD_MAX  (CSLD_SIZE >> 2)/* Number of Coarse-Second-Level Entries */
#define FSLD_MAX  (FSLD_SIZE >> 2)/* Number of Fine-Second-Level Entries */ 
#define LRG_CENTS 16 /* Number of Coarse-Second-Level Entries for a Large Page */
#define LRG_FENTS 64 /* Number of Fine-Second-Level Entries for a Large Page */
#define SML_FENTS 4 /* Number of Fine-Second-Level Entries for a Small Page */ 
/* Caching Attributes */

#define CB_CACHABLE   (1 << 3) /* Entry is Cachable */
#define CB_BUFFERABLE (1 << 2) /* Entry is Bufferable */
#define CB_IDEPENDENT (1 << 4) /* Implimentation Dependent */

/* Access Permitions */

#define AP_RESTRICTED 0 /* Supervisor: Restricted, User: Restricted (See ARM ARM) */
#define AP_NOACCESS   1 /* Supervisor: Read/Write, User: No Access */
#define AP_READACCESS 2 /* Supervisor: Read/Write, User: Read Only */ 
#define AP_ALLACCESS  3 /* Supervisor: Read/Write, User: Read/Write */

/* Domain Access Types */

#define DAT_NOACCESS 0 /* No Access (any access generates a domain fault) */
#define DAT_CLIENT   1 /* Client Domain (access permitions checked) */
#define DAT_RESERVED 2 /* Reserved Domain Type */
#define DAT_MANAGER  3 /* Manager Domain (no access permitions checks) */

#define KERNEL_DOMAIN 0 /* Kernel Domain */
#define USER01_DOMAIN 1 /* USER Domain (until FASS implimented) */
#define USER02_DOMAIN 2
#define USER03_DOMAIN 3
#define USER04_DOMAIN 4
#define USER05_DOMAIN 5
#define USER06_DOMAIN 6
#define USER07_DOMAIN 7
#define USER08_DOMAIN 8
#define USER09_DOMAIN 9
#define USER10_DOMAIN 10
#define USER11_DOMAIN 11
#define USER12_DOMAIN 12
#define USER13_DOMAIN 13
#define USER14_DOMAIN 14
#define USER15_DOMAIN 15

#ifdef LANGUAGE_C

typedef union dom_reg {
  word_t init;
  struct domains {
    word_t kernel:2;
    word_t user01:2;
    word_t user02:2;
    word_t user03:2;
    word_t user04:2;
    word_t user05:2;
    word_t user06:2;
    word_t user07:2;
    word_t user08:2;
    word_t user09:2;
    word_t user10:2;
    word_t user11:2;
    word_t user12:2;
    word_t user13:2;
    word_t user14:2;
    word_t user15:2;
  } domain;
} dom_reg_t;

#endif LANGUAGE_C

/********* 
* Macros * 
*********/

/* Virtual Address to FLD Index */
#define vaddrToFLD(vaddr) ((word_t)(vaddr) >> FLD_SHIFT)

/* FLD Index to Virtual Address */
#define fldToVaddr(fld) ((fld) << FLD_SHIFT)

/* Virtual Address to CSLD Index */
#define vaddrToCSLD(vaddr) (CSLD_MASK & ((word_t)(vaddr) >> CSLD_SHIFT))

/* CSLD Index to Partial Virtual Address */
#define csldToVaddr(csld) ((csld) << CSLD_SHIFT)

/* Physical Address to Section Base Address */
#define paddrToSBA(paddr) ((word_t)(paddr) >> FLD_SBA)

/* Section Base Address to Physical Address */
#define sbaToPaddr(sba) ((sba) << FLD_SBA)

/* Physical Address to Coarse Page Table Base Address */
#define paddrToCPTBA(paddr) ((word_t)(paddr) >> FLD_CPTBA)

/* Physical Address to Large Page Base Address */
#define paddrToLPBA(paddr) ((word_t)(paddr) >> SLD_LPBA)

/* Large Page Base Address to Physical Address */
#define lpbaToPaddr(lpba) ((lpba) << SLD_LPBA)

/* Physical Address to Small Page Base Address */
#define paddrToSPBA(paddr) ((word_t)(paddr) >> SLD_SPBA)

/* Small Page Base Address to Physical Address */
#define spbaToPaddr(spba) ((spba) << SLD_SPBA)

/* Coarse Page Table Base Address to Physical Address */
#define cptbaToPaddr(ptba) ((word_t)(ptba) << FLD_CPTBA)

/* Physical Address to Virtual Address (bank0 and kernel only) */
#define paddrToVaddr0(paddr) (RAM_VBASE | ((word_t)(paddr) - DRAM0_PBASE))

/* Virtual Address to Physical Address (bank0 and kernel only) */
#define vaddrToPaddr0(vaddr) ((~(RAM_VBASE) & (word_t)(vaddr)) + DRAM0_PBASE)


#endif MMU_H
