/*  Gauntlet/kernel/syscall.S
 *  Created by Adam Wiggins: 22/04/1999
 *  Last Modified by Adam Wiggins: 06/11/1999
 *  Generic ARM system call dispatch and entry points
 */

#include <asm.h>
#include <arm.h>
#include <ipc.h>
#include <tcb.h>
#include <types.h>
#include <kernel.h>
#include <macros.h>

/* soft_int_dispatch: Syscall dispatcher, invokes the relavent syscall.  
 * Pre:               SWI exception raised in user mode,
 *                    r0 = syscall number.
 * Post:              Syscall dispatched OR
 *                    Exception ipc generated to threads
 *                    exception handler thread if syscall was invalid,
 *                    Register set unmodifed.
 * Status:            Untested. */
EXCP(syscall_align)
.align 5				@ Alignment
	nop
	nop
	nop
	nop
	nop
EXCP(soft_int_dispatch)
	cmp	r0, #SYSCALL_LIMIT	@ Check syscall number is valid
	ldrlt	pc, [pc, r0, lsl #2]	@ Jump to syscall entry point
	b	invalid_syscall		@ Syscall >= SYSCALL_LIMIT	
.word	ipc_entry       	        @ Syscall = 0, ipc()
.word	id_nearest_entry	        @ Syscall = 1, id_nearest()
.word	fpage_unmap_entry       	@ Syscall = 2, fpage_unmap()
.word	thread_switch_entry     	@ Syscall = 3, thread_switch()
.word	thread_schedule_entry   	@ Syscall = 4, thread_schedule()
.word	lthread_ex_regs_entry   	@ Syscall = 5, lthread_ex_regs()
.word	task_new_entry          	@ Syscall = 6, task_new()
	nop				@ Align to fill DCache line
/* end soft_int_dispatch */

/* ipc_send_return: Return to thread from IPC send only syscall.
 * Pre:                 r0 = Caller Thread's Kernel Stack Pointer.
 * Post:                User sp, CPSR restored,
 *                      r3 = CC from send operation.
 * Registers:           r14 = Temp.
 * Status:              Untested. */
EXCP(ipc_send_return)
	POP(r0, r3)		@ Unstack CC 
        POP(r0, r14)		@ Unstack SPSR into Temp
/* S */ msr     spsr, r14	@ Load SPSR from Temp
        ldmfd   r0, {sp, pc}^	@ Restore User sp, CPSR and pc
/* end ipc_send_return */

/* ipc_entry: IPC entry point.
 * Pre:       r1 = Destination Thread's ID,
 *            r2 = Send Descriptor,
 *            r3 = Receive Descriptor,
 *            r4 = Timeouts,
 *            r5 - r12 = Short Message (to send),
 *            r13 = Kernel Misc Data Pointer.
 * Post:      r2 = Source Thread's ID,
 *            r3 = Message Dope + CC,
 *            r5 - r12 = Short Message (received), OR
 *            Caller stacked and polling or waiting on partner.
 * Registers: r0 = Current Thread's Kernel Stack Pointer,
 *            r1 = Destination Thread's Control Block,
 *            r2 = Temp/Current Thread's Control Block (receive phase),
 *            r13 = Temp,
 *            r14 = Temp/Current Thread's Control Block (send phase).
 * Status:    In progress. */
EXCP(ipc_entry)
	SYSCALL_STACK_USER()			@ Stack User State
/* Send operation */
	cmn	r2, #NEG_INVALID		@ Test for send operation
	beq	ipc_receive_only		@ If sndd != INVALID, go on
/* Test valid destination */
	TID2TCB(r14, r1)			@ Load up partners TCB into Temp
	cmn	r14, #NEG_KERNEL_TCB_CHECK	@ Check partner TCB range
	ldrb	r13, [r14, #T_COARSE_STATE]	@ Load partners coarse state
/* R */	bhs	no_partner			@ If Kernel TCB, no_partner
	teq	r13, #CS_UNUSED_TCB		@ Test if partner is alive
	beq	no_partner			@ if dead, no_partner
	ldr	r13, [r14, #T_MYSELF_TID]	@ Load up partners TID
/* S */	teq	r1, r13				@ Test if partner TID matches
	bne	no_partner			@ If version differs, no_partner
	mov	r1, r14				@ Move partner TCB into dtcb
	KSP2TCB(r14, r0)			@ Load callers TCB into stcb
/* Is partner ready to receive from caller? */
	ldrb	r13, [r1, #T_FINE_STATE]	@ Load partners fine state
/* S */	teq	r13, #FS_WAITING		@ Test if partner waiting
	bne	receiver_not_ready		@ If receiver ready, go on
	ldr	r13, [r1, #T_PARTNER]		@ Load who partners waiting on
/* S */	teq	r13, #NULL			@ Test for open wait
	teqne	r13, r14			@ If closed, Test waiting on us
	bne	receiver_not_ready		@ If open/waiting on us, go on
/* Is this a short/non-fpage/non-deceiving IPC? */
	teq	r2, #NULL			@ Test if short and fast send
	bne	not_short_send			@ If not, not_short_send
/* Short send */
GLOBAL(short_send)	
	cmn	r3, #NEG_INVALID		@ Test for receive operation
	bne	stack_after_send		@ If no receive, go on
GLOBAL(finish_send)				@ temp return point
	mov	r13, #IPC_OK			@ Load IPC_OK result
	PUSH(r0, r13)				@ Stack IPC CC result
	adr	r13, ipc_send_return		@ Load return address into Temp 
	PUSH(r0, r13)				@ Stack return procedure addr
	str	r0, [r14, #T_STACK_POINTER]	@ Store callers stack pointer
/* Switch to receivers context */
	mov	r13, #FS_READY			@ Set recevier to ready
	str	r13, [r1, #T_FINE_STATE]	@ Store receivers fine state
	LOAD_KDATA(r13)				@ Load Kernel Misc Data Pointer
	INSERT_READY_QUEUE(r1, r3, r4)		@ Add to ready queue, if removed
	DISPATCH_THREAD(r1, r14, r3)		@ Switch to receiver
/* Receive only IPC operation */
ipc_receive_only:
	cmn	r3, #NEG_INVALID		@ Test for receive operation
	beq	fast_syscall_return		@ If rcvd != INVALID, go on
/* Is receive open or closed? */
	tst	r3, #OPEN_RECEIVE		@ Test if receive is open
	bne	open_receive			@ If so, open_receive
/* Test valid sender */
	TID2TCB(r2, r1)				@ Load up partners TCB into Temp
	cmn	r2, #NEG_KERNEL_TCB_CHECK	@ Check partner TCB range
	ldrb	r14, [r2, #T_COARSE_STATE]	@ Load partners coarse state
/* R */	bhs	kernel_sender			@ If Kernel TCB, kernel_partner
	teq	r14, #CS_UNUSED_TCB		@ Test if partner is alive
	ldr	r5, [r2, #T_MYSELF_TID]		@ Load up partners TID
/* R */	beq	no_partner			@ if dead, no_partner
	teq	r1, r5				@ Test if partner TID matches
	bne	no_partner			@ If version differs, no_partner
	mov	r1, r2				@ Move partner TCB into dtcb
	KSP2TCB(r2, r0)				@ Load callers TCB into rtcb
/* Is partner ready to send to caller? */
	ldrb	r14, [r1, #T_FINE_STATE]	@ Load partners fine state
/* R */	ldr	r5, [r1, #T_PARTNER]		@ Load who partners polling
	teq	r14, #FS_POLLING		@ Test if partner polling
	bne	sender_not_ready		@ If sender ready, go on
	teq	r5, r2				@ Test if waiting on caller
	bne	sender_not_ready		@ If polling on caller, go on
/* Is this a short/non-fpage/non-deceiving IPC */
GLOBAL(do_receive)
	teq	r3, #NULL			@ Test if short and fast redeive
	bne	not_short_receive		@ If not, not_short_receive
/* Short receive */
GLOBAL(short_receive)
/* Remove sender from receivers send queue */
	ldr	r14, [r2, #T_SEND_START]	@ Load callers send queue head
/* R */	ldr	r5, [r2, #T_SEND_END]		@ Load callers send last in Temp
	teq	r14, r1				@ Test if sender is head
	ldr	r14, [r1, #T_SEND_NEXT]		@ Load senders send queue next
	ldr	r13, [r1, #T_SEND_PREV]		@ Load senders send queue prev
/* Sender is head of send queue */
	streq	r14, [r2, #T_SEND_START]	@ Callers start is senders next
/* Sender is not head of send queue */
	strne	r14, [r13, #T_SEND_NEXT]	@ Prevs next is senders next
	teq	r5, r1				@ Test if sender is last
/* Sender is end of send queue */
	streq	r13, [r2, #T_SEND_END]		@ Callers end is senders prev
/* Sender is not end of send queue */
	strne	r13, [r14, #T_SEND_PREV]	@ Nexts prev is senders prev	
/* Unstack stort message from sender */
	ldr	r3, [r1, #T_STACK_POINTER]	@ Load Pointer to message
/* R */	mov	r13, #IPC_OK			@ Set sender CC to return value
	ldm	r3!, {r5 - r12}			@ Restore short message
/* Stack senders return state */
	ldr	r14, [r3], #8			@ Unstack senders rcv descriptor
	cmn	r14, #NEG_INVALID		@ Test sender for receive
	bne	stack_after_receive		@ If not receiving, go on
GLOBAL(finish_receive)
	PUSH(r3, r13)				@ Stack ipc CC result
	adr	r14, ipc_send_return		@ Load return address into Temp
	PUSH(r3, r14)				@ Stack return procedure addr
	str	r3, [r1, #T_STACK_POINTER]	@ Store senders stack pointer
	mov	r14, #FS_WAITING		@ Set senders state to waiting
	str	r14, [r1, #T_FINE_STATE]	@ Store senders state
/* Set receivers return values */
	LOAD_KDATA(r13)				@ Load Kernel Misc Data Pointer
	INSERT_READY_QUEUE(r1, r3, r14)		@ Add to ready queue, if removed
	ldr	r2, [r1, #T_MYSELF_TID]		@ Load Senders TID into src
	mov	r3, #IPC_OK			@ Set receivers CC value
/* Return flows into fast_syscall_return */
/* end ipc_entry */

/* fast_syscall_return: Return to thread from syscall in .excp section.
 * Pre:                 r0 = Caller Thread's Kernel Stack Pointer,
 *                      All unbanked registers contain syscall return values.
 * Post:                User sp, CPSR restored,
 *                      returned from syscall with results in registers.
 * Registers:           r14 = Temp.
 * Status:              Done, Untested. */
EXCP(fast_syscall_return)
	POP(r0, r14)		@ Unstack SPSR into Temp
/* S */	msr     spsr, r14	@ Load SPSR from Temp
	ldmfd   r0, {sp, pc}^	@ Restore User sp, CPSR and pc
/* end fast_syscall_return */

/* id_nearest: id_nearest entry point.
 * Pre:        r1 = Destination TID,
 *             r13 = Kernel Misc Data Pointer.
 * Post:       r1 = Type,
 *             r2 = Nearest ID
 * Registers:  r0 = Current Thread's Kernel Stack Pointer,
 *             r7 = Current Thread's TCB,
 *             r14 = Temp.
 * Status:     get_My_ID() implimented only, 
 *             Untested.
 * Priority:   Finishing is extreamly low priority since Clan's & Chief's is to
 *             be replaced, but I will implimented for reference. */
EXCP(id_nearest_entry)
	SYSCALL_STACK_USER()			@ Stack User State
/* Test for get_My_ID case */
	teq	r1, #NIL_TID			@ Test if Dest NIL Thread
	bne	nearest_chief			@ If not find Nearest ID
        KSP2TCB(r7, r0)				@ Get current TCB from ksp
/* Get threads own TID */
	ldr	r2, [r7, #T_MYSELF_TID]		@ Load threads TID into nid
/* Return */
	b	fast_syscall_return
nearest_chief:					@ Find cheif of dest id
/* TEMP */
	mov	r2, #INVALID_TID		@ return invalid TID
/* end TEMP start of real stuff */
//	KSP2CHIEF(ctc, cksp)			@ Load up our chief
//	TID2CHIEF(dtc, dst)			@ Load up destinations chief
/* Check if chief is the same */
	// FILL ME (if so type is same, return dest's id)
/* Check if destination is chief */
	// FILL ME (if so type is outer, return dest's id)
1:						@ Loop until nearest chief found
/* Check if we are chief of dest */
	// FILL ME (if so type is inner, return dest's id)
/* Check if chief is the same */
	// FILL ME (if so type is inner, return dest's id)
/* Check if our depth is the same */
	// FILL ME (if so type is outer, return our chief's id)
/* Return */
	b	fast_syscall_return
/* end id_nearest_entry */

/* fpage_unmap_entry: fpage_unmap entry point.
 * Status:            Not Done, acts as a nop for now.
 * Priority:          This will be done once the mapping database works. */
EXCP(fpage_unmap_entry)
	SYSCALL_STACK_USER()	@ Stack User State
	/* FILL ME! */
/* Return */
        b	fast_syscall_return
/* end fpage_unmap_entry */

/* thread_switch_entry: thread_switch entry point.
 * Pre:                 r1 = Destination TID,
 *                      r13 = Kernel Misc Data Pointer.
 * Post:                destination thread dispatched if ready, else next ready
 *                      thread dispatched.
 * Registers:           r0 = Current Thread's Kernel Stack Pointer,
 *                      r1 = Destination Thread's Control Block,
 *                      r7 = Current Thread's Control Block,
 *                      r12 = Temp,
 *                      r14 = Temp.
 * Status:              Done, Untested. Should test version numbers! */
EXCP(thread_switch_entry)
	SYSCALL_STACK_USER()			@ Stack User State
	adr	r14, fast_syscall_return	@ Set return procedure address
	PUSH(r0, r14)				@ Stack return procedure address
	str	r0, [r7, #T_STACK_POINTER]	@ Store threads stack pointer
/* Test if destination thread specified */
	teq	r1, #NIL_TID			@ Test if NIL thread specified
	beq	dispatch_next			@ If NIL_TID dispatch_next
GLOBAL(thread_switch)				@ Label for in kernel calls
/* Test destination thread is ready to run */
	TID2TCB(r1, r1)				@ Load destination TCB into dtcb
	ldrb	r14, [r1, #T_COARSE_STATE]	@ Load dest threads coarse state
/* R */	ldrb	r12, [r1, #T_FINE_STATE]	@ Load dest threads fine state
	teq	r14, #CS_UNUSED_TCB		@ Test if thread is active
	beq	dispatch_next			@ If inactive dispatch_next
	teq	r12, #FS_READY			@ Test if thread is ready/busy
	bne	dispatch_next			@ If busy dispatch_next
/* Dispatch destination thread */
	b	dispatch_thread
/* end thread_switch_entry */

/* thread_schedule_entry: thread_schedule entry point.
 * Status:                Not Done, acts as a nop for now.
 * Priority:              Do after thread_switch(). */
EXCP(thread_schedule_entry)
	SYSCALL_STACK_USER()	@ Stack User State
	/* FILL ME! */
/* Return */
        b	fast_syscall_return
/* end thread_schedule_entry */

/* lthread_ex_regs_entry: lthread_ex_regs entry point.
 * Pre:                   r1 = lthread no,
 *                        r2 = New SP,
 *                        r3 = New PC,
 *                        r4 = New Pager Thread ID, 
 *                        r5 = New Exception Handler Thread ID,
 *                        r13 = Kernel Misc Data Pointer.
 * Post:                  r1 = Old SP,
 *                        r2 = Old PC,
 *                        r3 = Old Pager Thread ID,
 *                        r4 = Old Exception Handler Thread ID
 *                        r5 = Old CPSR.
 * Registers:             r0 = Current Thread's Kernel Stack Pointer,
 *                        r6 = lthread's Kernel Stack Pointer,
 *                        r7 = Current Thread's Control Block, 
 *                        r9 = lthread's Control Block,
 *                        r12 = Temp 0,
 *                        r14 = Temp 1.
 * Status:                Only thread create correct. */
EXCP(lthread_ex_regs_entry)
	SYSCALL_STACK_USER()			@ Stack User State
/* Get TCB of lthread */
	KSP2TCB(r7, r0)				@ Get Caller Threads TCB address
	bic	r9, r7, #TCB_THREAD_MASK	@ Mask Thread number from TCB
	orr	r9, r9, r1			@ Generate lthread TCB address
/* Is TCB used? */
	ldrb	r12, [r9, #T_COARSE_STATE]	@ Load lthreads coarse state
/* R */	ldrb	r14, [r9, #T_FINE_STATE]	@ Load lthreads fine state
	teq	r12, #CS_UNUSED_TCB		@ Test if tcb is active
	beq	create_thread			@ If unused, create_thread
/* Exchange registers of an existing thread */
	teq	r14, #FS_READY			@ Test if in kernel or not
	bne	kernel_ex_regs			@ If in kernel, kernel_ex_regs
/* lthread is in User Mode */
	ldr	r6, [r9, #T_STACK_POINTER]	@ Load lthreads kernel SP
/* S */	ldr	r1, [r6, #SSL_USP]		@ Load old_SP
	cmn	r2, #NEG_INVALID		@ Test SP for replacement
	strne	r2, [r6, #SSL_USP]		@ If replace, store new_SP 
	ldr	r2, [r6, #SSL_PC]		@ Load old_PC
	cmn	r3, #NEG_INVALID		@ Test PC for replacement
	strne	r3, [r6, #SSL_PC]		@ If replace, store new_PC
	b	rest_ex_regs			@ Finish up exchanging
/* lthread is in a Kernel Mode */
kernel_ex_regs:
	// FILL ME
/* Finish off the common exchanges */
rest_ex_regs:					
	ldr	r3, [r9, #T_PAGER_TID]		@ Load old_pager_tid
	cmn	r4, #NEG_INVALID		@ Test pager for replacement
	strne	r4, [r9, #T_PAGER_TID]		@ if replace,store new_pager_tid
	ldr	r4, [r9, #T_EXCPT_TID]		@ Load old_excpt_tid
	cmn	r5, #NEG_INVALID		@ Test excpt for replacement
	strne	r5, [r9, #T_EXCPT_TID]		@ If replace,store new_excpt_tid
/* Return */
        b	fast_syscall_return
/* Create (Activate) lthread */
create_thread:
/* Generate and set lthreads TID */
	ldr	r12, [r7, #T_MYSELF_TID]	@ Load Callers TID into Temp
/* R */	mov	r6, r9				@ Load ltcb address into lksp
	bic	r12, r12, #TID_THREAD_MASK	@ Mask thread number from TID
	orr	r12, r12, r1			@ Generate lthread TID
	str	r12, [r9, #T_MYSELF_TID]	@ Store lthreads TID in its TCB
/* Set up lthreads kernel stack */ 
	and	r6, r6, #TCB_STACK_MASK		@ Initalise stack pointer
/* Set SP and PC */
	PUSH(r6, r3)				@ Stack user_pc in kernel stack
	PUSH(r6, r2)				@ Stack user_sp in kernel stack
/* Set pager and exception handler TIDs */
	str	r4, [r9, #T_PAGER_TID]		@ Store pager_tid in TCB
	str	r5, [r9, #T_EXCPT_TID]		@ Store excpt_tid in TCB
/* Generate and set CPSR */
	mov	r5, #USR_MODE			@ Set user mode in Temp 0
	mov	r5, #(FIQ_MASK | IRQ_MASK)	@ Mask interrupts (for now)
	PUSH(r6, r5)				@ Stack SPSR in kernel stack 
/* Set return handler to fast_syscall_return */
	adr	r12, fast_syscall_return	@ Load address into Temp 0
	PUSH(r8, r12)				@ Stack return handler address
        str	r6, [r9, $T_STACK_POINTER]	@ Store lthreads kernel stack
/* Initialise lthreads send queue */
	mov	r12, #NULL			@ Move NULL to temp
	str	r12, [r9, #T_SEND_END]		@ Set send queue to empty
/* Mark ltcb as active and runable */
	mov	r12, #CS_THREAD_ACTIVE		@ Set lthread as active
	strb	r12, [r9, #T_COARSE_STATE]	@ Store coarse state in TCB
	mov	r12, #FS_READY			@ Set lthread as ready
	strb	r12, [r9, #T_FINE_STATE]	@ Store fine state in TCB
/* Set lthread mcp & priority */
	ldrb	r14, [r7, #T_PRIORITY]		@ Load caller threads priority
/* R */	ldrb	r12, [r7, #T_MCP]		@ Load caller threads mcp
   	strb	r14, [r9, #T_PRIORITY]		@ Store priority in lthreads TCB
	strb	r12, [r9, #T_MCP]		@ Store mcp in lthreads TCB
/* Set lthreads page directory */
	ldr	r14, [r7, #T_PAGE_DIR]		@ Load callers page directory
/* S */ str	r14, [r9, #T_PAGE_DIR]		@ Store page dir in lthreads TCB
/* Add lthread to relevant ready queue */
	INSERT_READY_QUEUE(r9, r12, r14)
/* Add lthread to the present list */
/* R */	ldr	r12, [r7, #T_PRESENT_NEXT]	@ Load next present thread
	str	r7, [r9, #T_PRESENT_PREV]	@ Point back to caller thread
	str	r12, [r9, #T_PRESENT_NEXT]	@ Point forward to next thread
	str	r9, [r7, #T_PRESENT_NEXT]	@ Updated caller threads next
	str	r9, [r12, #T_PRESENT_PREV]	@ Update next threads prev
/* Return */
	b	fast_syscall_return
/* end lthread_ex_regs_entry */

/* task_new_entry: task_new entry point.
 * Pre:                   r1 = Destination Task ID,
 *                        r2 = MCP/ New Chief,
 *                        r3 = SP,
 *                        r4 = PC,
 *                        r5 = Pager Thread ID,
 *                        r6 = Exception Handler Thread ID,
 *                        r13 = Kernel Misc Data Pointer.
 * Post:                  r1 = New Task ID.
 * Registers:             r0 = Current Thread's Kernel Stack Pointer,
 *                        r1 = Destination's Thread Control Block,
 *                        r8 = Destination's Kernel Stack Pointer,
 *                        r7 = Current Thread's Control Block,
 *                        r12 = Temp 0,
 *                        r14 = Temp 1.
 * Status:         In progress. */
EXCP(task_new_entry)
	SYSCALL_STACK_USER()	@ Stack User State
//	TID2TCB(r1, r1)		@ Get destinations TCB
	/* FILL ME! */
/* Return */
        b	fast_syscall_return
/* end task_new_entry */

/* invalid_syscall: Handle invalid syscall to kernel.
 * Post:            invalid syscall exception ipc sent to threads exception 
 *                  handler, else exception handler isn't valid terminate the
 *                  thread.
 * Status:	    Not done, requires exception ipc, acts as a nop for now.
 * Priority         Not very important. Just act as a nop for now. */
EXCP(invalid_syscall)
	SYSCALL_STACK_USER()	@ Stack User State (Temporary)
	/* FILL ME! */
/* Return */
	b	fast_syscall_return
/* end invalid_syscall */

/* should put fastest of 'slow' syscalls here so a branch can be avoided like
 * with ipc and fast_syscall_return.
 * Note: A slow syscall is one that has code in the .text section. */

/* syscall_return: Return to thread from syscall in .text section.
 * Pre:            r0 = Caller Thread's Stack Pointer,
 *                 All unbanked registers contain syscall return values,
 * Post:           User sp and CPSR restored,
 *                 returned from syscall with results in registers.
 * Registers:      r14 = Temp.
 * Status:         Done, Untested. */
PROC(syscall_return)
	POP(r0, r14)			@ Unstack SPSR into Temp
/* S */	msr     spsr, r14		@ Load SPSR from Temp
	ldmfd   r0, {sp, pc}^		@ Restore User sp, CPSR and pc
/* end syscall_return */
