/*  Gauntlet/kernel/tcb.h
 *  Created by Adam Wiggins: 27/10/1999
 *  Last Modified by Adam Wiggins: 06/11/1999
 *  Generic ARM Thread Control Block (TCB) macros and defines
 */

#ifndef TCB_H
#define TCB_H

#include <types.h>
#include <kernel.h>

#define TCB_SIZE_LOG2   8 /* TCB's 256 bytes */
#define TCB_SIZE        (1 << TCB_SIZE_LOG2)
#define TCB_STACK_MASK  (TCB_SIZE - 1)
#define TCB_THREAD_MASK (TID_THREAD_MASK << TCB_SIZE_LOG2)
#define TCB_TASK_SHIFT  (TCB_SIZE_LOG2 + MAX_THREADS_LOG2)

/************
* Thread ID *
************/

#define TID_VERSION_MASK 0xFFFC0000 /* 14 bits, 31-18 */
#define TID_TASK_MASK    0x0003FF00 /* 10 bits, 17-8 */
#define TID_THREAD_MASK  0x000000FF /* 8 bits, 7-0 */

#ifdef LANGUAGE_C

typedef union threadid {
  word_t init;
  struct {
    word_t task:10;
    word_t thread:8;
    word_t version:14;
  } id;
} threadid_t;

#endif LANGUAGE_C

/*********************************
* Thread Control Block Structure *
*********************************/

/* NOTE: If the TCB data structure is changed the initialisation code for 
 *       initSigma0TCB() and the lthread_ex_regs()/task_new() syscall code 
 *       must be updated for the change. (Maybe more?) */

/* Task State */ 
#define T_PAGE_DIR      0 /* 4, Page Directory Pointer */
#define T_MCP           4 /* 1, Maximum Controled Priority */

/* Thread State */
#define T_PRIORITY      5  /* 1, Thread's Priority */
#define T_COARSE_STATE  6  /* 1, Thread's Coarse_state */
#define T_FINE_STATE    7  /* 1, Thread's Fine_state */
#define T_MYSELF_TID    8  /* 4, Thread's ID */
#define T_PAGER_TID     12 /* 4, Thread's Pager TID */
#define T_EXCPT_TID     16 /* 4, Thread's Exception Handler TID */
#define T_STACK_POINTER 20 /* 4, Kernel Stack */

/* IPC State */
#define T_TIMEOUTS    24 /* 4, IPC Timeouts value */
#define T_PARTNER     28 /* 4, Who thread is polling or waiting on */
#define T_SEND_START  32 /* 4, Thread's Send Queue (threads polling this tcb) */
#define T_SEND_END    36 /* 4, End of Thread's send Queue */

/* Kernel Links */
#define T_SEND_NEXT    40 /* 4, Next Send Thread (send queue of partner) */
#define T_SEND_PREV    44 /* 4, Prev Send Thread (send queue of partner) */
#define T_PRESENT_NEXT 48 /* 4, Next Present Thread */
#define T_PRESENT_PREV 52 /* 4, Prev Present Thread */
#define T_READY_NEXT   56 /* 4, Next Ready Thread (Of Thread's Priority) */
#define T_READY_PREV   60 /* 4, Prev Ready Thread (Of Thread's Priority) */

/* TCB Magic number 13579BDF (easy to check stack overrun) */
#define T_MAGIC 64

/* End of TCB */
#define T_END 68

#ifdef LANGUAGE_C

typedef struct tcb {
  union fld      *page_dir;
  byte_t         mcp;
  byte_t         priority;
  byte_t         coarse_state;
  byte_t         fine_state;
  union threadid myself_tid;
  union threadid pager_tid;
  union threadid excpt_tid;
  word_t         *stack_pointer;
  word_t         timeouts;
  struct tcb     *partner;
  struct tcb     *send_start;
  struct tcb     *send_end;
  struct tcb     *send_next;
  struct tcb     *send_prev;
  struct tcb     *present_next;
  struct tcb     *present_prev;
  struct tcb     *ready_next;
  struct tcb     *ready_prev;
  word_t         magic;
  byte_t         pad[188];
} tcb_t;

#endif LANGUAGE_C
     
/* Coarse States */

#define CS_UNUSED_TCB    0x0
#define CS_THREAD_ACTIVE 0x1

/* Fine States */

#define FS_READY   0x0
#define FS_WAITING 0x1
#define FS_POLLING 0x2

/****************
* Stack Layouts *
****************/

/* Syscall Stack Layout (Top of stack only) */

#define SSL_PC   12
#define SSL_USP  8
#define SSL_SPSR 4
#define SSL_PLR  0

/* Preemption Stack Layout (Top of stack only) */

#define PSL_PC   68
#define PSL_ULR  64
#define PSL_USP  60
#define PSL_R12  56
#define PSL_R11  52
#define PSL_R10  48
#define PSL_R9   44
#define PSL_R8   40
#define PSL_R7   36
#define PSL_R6   32
#define PSL_R5   28
#define PSL_R4   24
#define PSL_R3   20
#define PSL_R2   16
#define PSL_R1   12
#define PSL_R0   8
#define PSL_SPSR 4
#define PSL_PLR  0

/*********
* Macros *
*********/

/* TID2TCB(tcb, tid): Return the address of the TCB for the given TID.
 * Pre:               tid = contains a valid TID (ie non-kernel tid).
 * Post:              tcb = address of TID's TCB,
 *                    tid unchanged. */
#define TID2TCB(tcb, tid)                                                      \
        mov     tcb, tid, lsl $TCB_SIZE_LOG2                                  ;\
        orr     tcb, tcb, $TCBA_VBASE
/* end TID2TCB() */

/* KSP2TCB(tcb, ksp): Return the address of the TCB for the given kernel stack.
 * Pre:               ksp = A non-empty Kernel Stack (empty points to next tcb).
 * Post:              tcb = Address of Kernel Stack Pointers TCB,
 *                    ksp unchanged. */
#define KSP2TCB(tcb, ksp)                                                      \
        and     tcb, ksp, $TCB_STACK_MASK
/* end KSP2TCB() */

/* Thread ID to TCB Index */
#define tidToTCB(tid) (((word_t)(tid) & ~TID_VERSION_MASK)) 


#endif TCB_H
